% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metaRVM.R
\name{metaRVM}
\alias{metaRVM}
\title{Run a MetaRVM epidemic simulation}
\usage{
metaRVM(config_input)
}
\arguments{
\item{config_input}{Configuration specification in one of three forms:
\itemize{
\item \strong{Character string}: path to a YAML configuration file.
\item \strong{\code{\link{MetaRVMConfig}} object}: pre-initialized configuration.
\item \strong{Named list}: output from \code{\link[=parse_config]{parse_config()}} with
\code{return_object = FALSE}.
}}
}
\value{
A \code{\link{MetaRVMResults}} R6 object with three key components:
\describe{
\item{$results}{A tidy \code{data.table} with one row per
date–subpopulation–disease state–instance combination. Typical
columns include:
\itemize{
\item \code{date}: calendar date (\code{Date})
\item \code{age}, \code{race}, \code{zone}: demographic categories (if present
in the population mapping)
\item \code{disease_state}: compartment or flow label (e.g., \code{S}, \code{E},
\code{I_symp}, \code{H}, \code{R}, \code{D}, \code{n_SE}, \code{n_IsympH}, etc.)
\item \code{value}: population count or daily flow
\item \code{instance}: simulation instance index (1, 2, …)
}
}
\item{$config}{The \code{\link{MetaRVMConfig}} object used for the run.}
\item{$run_info}{A list with metadata such as \code{n_instances},
\code{date_range}, \code{delta_t}, and checkpoint information.}
}
}
\description{
\code{metaRVM()} is the high-level entry point for running a MetaRVM
metapopulation respiratory virus simulation. It parses the configuration,
runs one or more simulation instances (deterministic or stochastic),
formats the ODIN/MetaRVM output into a tidy long table with calendar
dates and demographic attributes, and returns a \code{\link{MetaRVMResults}}
object for downstream analysis and plotting.
}
\details{
The configuration input controls:
\itemize{
\item \strong{Population structure} (e.g., age, race, zone)
\item \strong{Disease parameters} (\code{ts}, \code{tv}, \code{ve}, \code{de}, \code{dp}, \code{da}, \code{ds},
\code{dh}, \code{dr}, \code{pea}, \code{psr}, \code{phr}, \code{dv}, etc.)
\item \strong{Mixing matrices} (weekday/weekend, day/night contact patterns)
\item \strong{Vaccination schedule} and immunity waning
\item \strong{Simulation settings} (start date, length, number of instances,
stochastic vs. deterministic mode, checkpointing)
}

Internally, \code{metaRVM()}:
\enumerate{
\item Parses the YAML configuration via \code{\link[=parse_config]{parse_config()}}.
\item Calls the ODIN-based simulation engine \code{\link[=meta_sim]{meta_sim()}} for each instance.
\item Uses \code{\link[=format_metarvm_output]{format_metarvm_output()}} to convert time steps to dates and
attach demographic attributes.
\item Wraps the formatted output and metadata in a \code{\link{MetaRVMResults}}
object that supports method chaining for subsetting, summarizing,
and plotting.
}
}
\examples{
\donttest{
options(odin.verbose = FALSE)
example_config <- system.file("extdata", "example_config.yaml",
                              package = "MetaRVM")

# Run a single-instance simulation from a YAML file
results <- metaRVM(example_config)

# Print a high-level summary
results

# Access the tidy results table
head(results$results)

# Summarize and plot hospitalizations and deaths by age and race
results$summarize(
  group_by       = c("age", "race"),
  disease_states = c("H", "D"),
  stats          = c("median", "quantile"),
  quantiles      = c(0.25, 0.75)
)$plot()

# Using a pre-parsed configuration object
cfg <- parse_config(example_config, return_object = TRUE)
results2 <- metaRVM(cfg)
}
}
\references{
Fadikar, A., et al. "Developing and deploying a use-inspired metapopulation modeling framework for detailed tracking of stratified health outcomes"
}
\seealso{
\code{\link[=parse_config]{parse_config()}} for reading YAML configurations,
\link{MetaRVMConfig} for configuration management,
\link{MetaRVMResults} for analysis and plotting,
\code{\link[=meta_sim]{meta_sim()}} for the low-level simulation engine.
}
\author{
Arindam Fadikar, Charles Macal, Ignacio Martinez-Moyano, Jonathan Ozik
}
