#' Predict method for objects of class mff
#'
#' @description
#' The \emph{predict} method for objects of class \emph{mff} is used to generate predictions from the Meta
#' Fuzzy Function framework on the test dataset.
#'
#' @param object an object of class mff (returned by mff() or tune.mff()).
#' @param pred_matrix numeric matrix (\eqn{N_{test} \times M}) of base predictions for the new data (e.g.,
#' test set).
#' @param type 'best' uses the optimal configuration/cluster selected by tune.mff(); 'all' returns outputs
#' for all tuned configurations.
#' @param ... further arguments passed to or from other methods.
#'
#' @details
#' In this setting, the input to \emph{predict} is the matrix of test-set predictions produced by the base
#' models that were trained in the earlier stage using \emph{model.train}.
#'
#' Let \eqn{X_{test} \in \mathbb{R}^{N_{test} \times M}}{X_test \in R^{N_{test} \times M}} denote the prediction matrix of \eqn{M} trained base models on the test dataset,
#' where \eqn{N_{test}} is the number of test observations. Each column of \eqn{X_{test}} corresponds to the predictions
#' generated by a single base learner. Using the membership-based weight matrix \eqn{W = [w_{mc}]} obtained
#' during validation, meta fuzzy function predictions on the test set are computed as
#'
#' \deqn{\hat{y}_{MFF,test}^{(c)} = \sum_{m=1}^{M} w^{(m)(c)} \hat{y}_{test}^{(m)} \quad (11)}{y_hat_{MFF,test}^{(c)} = sum_{m=1}^{M} w^{(m)(c)} y_hat_{test}^{(m)} (11)}
#'
#' where \eqn{\hat{y}_{test}^{(m)}} denotes the test-set prediction vector produced by base model \eqn{m}, and \eqn{c = 1, \dots, C} indexes
#' the meta fuzzy functions.
#'
#' The predict function supports two prediction modes, controlled by the type argument:
#' \itemize{
#'   \item type = "best" returns predictions obtained from the single meta fuzzy function that achieved
#'   the optimal validation performance during training or tuning (see Eq. 9).
#'   \item type = "all" returns predictions from all meta fuzzy functions, allowing users to examine
#'   alternative predictive components within the MFF model.
#' }
#'
#' When type = "best" is selected, the final prediction for test observation \eqn{i} is given by
#'
#' \deqn{\hat{y}_{i,final} = \sum_{m=1}^{M} w^{(m)(c^*)} \hat{y}_{i,test}^{(m)} \quad (12)}{\hat{y}_{i,final} = \sum_{m=1}^{M} w^{(m)(c^*)} \hat{y}_{i,test}^{(m)} (12)}
#'
#' where \eqn{c^*} denotes the index of the best-performing meta fuzzy function selected based on validation
#' performance.
#'
#' Computes cluster-wise predictions via membership-weighted aggregation of base-model
#' predictions. For inspection and reporting purposes, a rounded copy of the membership
#' weights (four decimal places) is returned.
#'
#' If a single best-performing cluster is defined via tune.mff, the corresponding weight vector is
#' also returned.
#'
#' @return
#' \itemize{
#'   \item \code{mff_preds}: A numeric matrix of meta fuzzy function predictions on the test set. When
#'   "best", this matrix reduces to a single column.
#'  \item \code{mff_weights}: The membership based weight matrix. If a single meta fuzzy function is
#'  selected, the corresponding weight vector is returned.
#' }
#'
#'
#' @seealso \code{\link{mff}}, \code{\link{tune.mff}}, \code{\link{evaluate}}
#'
#' @examples
#'   res <- model.train(target="medv", data=MASS::Boston, ntest=50, nvalid=50, seed = 123)
#'   fit <- tune.mff(res$pred_matrix_valid, res$y_valid, max_c=6, mff.method="kmeans")
#'   out <- predict(fit, pred_matrix=res$pred_matrix_test, type="best")
#'   head(out$mff_preds)
#'   out$mff_weights
#'
#' @export
predict.mff <- function(object, pred_matrix, type = c("best", "all"), ...) {

  type <- match.arg(type)

  if (type == "best") {

    weights <- if (!is.null(object$best_weight)) {
      object$best_weight
    } else {
      object$weights
    }

  } else {
    weights <- object$weights
  }

  preds <- pred_matrix %*% weights

  list(
    mff_preds   = preds,
    mff_weights = round(weights, 4)
  )
}

