#'
#' Load the Scaler for the pre-trained Long Short Term Memory (LSTM) Network
#'
#' @description
#' Loads the scaler object within the \code{LSTMfactors} package. This object
#' is a \code{list} containing a mean vector and
#' a standard deviation vector, which were computed from the 1,000,000
#' datasets \code{\link[LSTMfactors]{data.datasets.LSTM}}
#' training the Long Short Term Memory (LSTM) Network. It
#' serves as a tool for normalizing features in
#' \code{\link[LSTMfactors]{LSTM}}.
#'
#'
#' @seealso \code{\link[LSTMfactors]{LSTM}}, \code{\link[LSTMfactors]{normalizor}}, \code{\link[LSTMfactors]{data.scaler.LSTM}}
#'
#' @return scaler objective.
#'
#' @examples
#' library(LSTMfactors)
#'
#' scaler <- load.scaler()
#' print(scaler)
#'
#'
#' @export
#'
load.scaler <- function() {
  scaler.path <- system.file("data", paste0("data.scaler.", "LSTM",".rda"), package = "LSTMfactors")

  local.env <- new.env()
  load(scaler.path, envir = local.env)
  data.scaler <- get(ls(local.env)[1], envir = local.env)

  return(data.scaler)
}

#'
#' Load the pre-trained Long Short Term Memory (LSTM) Network
#'
#' @description
#' Loads the pre-trained Long Short Term Memory (LSTM) Network form \code{LSTM.onnx}.
#' The function uses the \code{reticulate} package to import the \code{onnxruntime} Python library
#' and create an inference session for the model.
#' @seealso \code{\link[LSTMfactors]{LSTM}}
#'
#'
#' @return An ONNX runtime inference session object for the LSTM model.
#'
#' @note
#' Note that Python (suggested >= 3.11) and the libraries \code{numpy} and \code{onnxruntime} are required.
#'
#' First, please ensure that Python is installed on your computer and that Python is
#' included in the system's PATH environment variable. If not,
#' please download and install it from the official website (https://www.python.org/).
#'
#' If you encounter an error when running this function stating that the \code{numpy} and \code{onnxruntime}
#' modules are missing:
#'
#'  \code{Error in py_module_import(module, convert = convert) :}
#'
#'    \code{ModuleNotFoundError: No module named 'numpy'}
#'
#' or
#'
#'  \code{Error in py_module_import(module, convert = convert) :}
#'
#'    \code{ModuleNotFoundError: No module named 'onnxruntime'}
#'
#' this means that the \code{numpy} or \code{onnxruntime} library is missing from your Python environment. If you are using Windows or macOS,
#' please run the command \code{pip install numpy} or \code{pip install onnxruntime} in Command Prompt or Windows PowerShell (Windows), or Terminal (macOS).
#' If you are using Linux, please ensure that \code{pip} is installed and use the command \code{pip install numpy} or
#' \code{pip install onnxruntime} to install the missing libraries.
#'
#'
#' @export
#' @import reticulate
#' @importFrom reticulate import
#'
load.LSTM <- function() {
  NN.path <- system.file("extdata", paste0("LSTM", ".onnx"), package = "LSTMfactors")
  onnxruntime <- import("onnxruntime")
  NN <- onnxruntime$InferenceSession(NN.path)

  return(NN)
}
