# =============================================================================
# KWELA 1.0.0 SUMMARY & DIAGNOSTICS
# =============================================================================

#' Treatment-Level Summary
#'
#' Extracts treatment-level consensus results from kwela_analyze output.
#'
#' @param result Output from kwela_analyze
#' @return Data frame with treatment-level classification summary
#' @export
#' @examples
#' \donttest{
#' set.seed(42)
#' df <- data.frame(
#'   Treatment = c(rep("Positive Control", 8), rep("Negative Control", 8),
#'                 rep("Sample_A", 8)),
#'   TTT = c(rnorm(8, 8, 1), rnorm(8, 72, 5), rnorm(8, 12, 3)),
#'   MP = c(rnorm(8, 100, 10), rnorm(8, 20, 5), rnorm(8, 85, 15))
#' )
#' result <- kwela_analyze(df, verbose = FALSE)
#' summary <- kwela_summarize(result)
#' }
kwela_summarize <- function(result) {
  trt_summary <- attr(result, "trt_summary")
  if (is.null(trt_summary)) {
    stop("No treatment summary found. Run kwela_analyze() first.")
  }
  trt_summary
}

#' Diagnostics for KWELA Analysis
#'
#' Returns detailed diagnostic information including separation quality,
#' optimal cutoffs, and well/treatment-level statistics.
#'
#' @param result Output from kwela_analyze
#' @return List containing version, profile, consensus rule, optimal cutoff,
#'   Youden J statistic, separation metrics, and well/treatment-level summaries
#' @export
#' @examples
#' \donttest{
#' set.seed(42)
#' df <- data.frame(
#'   Treatment = c(rep("Positive Control", 8), rep("Negative Control", 8),
#'                 rep("Sample_A", 8)),
#'   TTT = c(rnorm(8, 8, 1), rnorm(8, 72, 5), rnorm(8, 12, 3)),
#'   MP = c(rnorm(8, 100, 10), rnorm(8, 20, 5), rnorm(8, 85, 15))
#' )
#' result <- kwela_analyze(df, verbose = FALSE)
#' diag <- kwela_diagnostics(result)
#' }
kwela_diagnostics <- function(result) {
  pc_stats <- attr(result, "pc_stats")
  nc_stats <- attr(result, "nc_stats")
  separation <- attr(result, "separation")
  samples <- result[result$Type == "Sample", ]
  spiked <- samples[samples$is_spiked, ]
  unspiked <- samples[!samples$is_spiked, ]

  pos_cats <- c("POSITIVE")

  # Instability summary
  instab <- attr(result, "instability_summary")
  n_matrix_unstable <- if (!is.null(instab)) sum(vapply(instab, function(x) x$unstable, logical(1))) else 0
  treatments_flagged <- if (!is.null(instab)) names(instab)[vapply(instab, function(x) x$unstable, logical(1))] else character(0)

  list(
    version = attr(result, "version"),
    mode = attr(result, "mode"),
    profile = attr(result, "profile"),
    consensus = attr(result, "consensus"),
    optimal_cutoff = attr(result, "optimal_cutoff"),
    youden_j = attr(result, "youden_j"),
    separation = separation,
    well_level = list(
      spiked_recovery = if (nrow(spiked) > 0)
        sum(spiked$classification %in% pos_cats) / nrow(spiked) else NA,
      unspiked_fp = if (nrow(unspiked) > 0)
        sum(unspiked$classification %in% pos_cats) / nrow(unspiked) else NA,
      n_artifact = sum(samples$artifact_flag, na.rm = TRUE)
    ),
    treatment_level = list(
      classification_table = table(attr(result, "trt_summary")$classification)
    ),
    instability = list(
      check_enabled = attr(result, "instability_check"),
      strictness = attr(result, "instability_strictness"),
      n_inconclusive_matrix = n_matrix_unstable,
      treatments_flagged = treatments_flagged
    ),
    n_samples = nrow(samples),
    n_spiked = nrow(spiked),
    n_unspiked = nrow(unspiked)
  )
}
