# =============================================================================
# KWELA 1.0.0: 6-LAYER HIERARCHICAL ADAPTIVE CLASSIFICATION
# Dual-mode + instability detection + group-aware mixed-assay support
# =============================================================================

# --- RAF Scoring Functions (internal) ---

#' @noRd
raf_ratio_score <- function(raf, mp, pc_ratios, nc_ratios) {
  if (!is.finite(raf) || !is.finite(mp) || mp <= .Machine$double.eps)
    return(0.5)
  ratio <- raf / mp
  all_ratios <- c(pc_ratios, nc_ratios)
  all_ratios <- all_ratios[is.finite(all_ratios)]
  if (length(all_ratios) < 3) return(0.5)
  ratio_sd <- stats::sd(all_ratios)
  if (!is.finite(ratio_sd) || ratio_sd < .Machine$double.eps) return(0.5)
  stats::ecdf(all_ratios)(ratio)
}

#' @noRd
raf_zscore <- function(raf, nc_raf_median, nc_raf_mad) {
  if (!is.finite(raf) || !is.finite(nc_raf_median) || !is.finite(nc_raf_mad) ||
      nc_raf_mad <= .Machine$double.eps) return(0.5)
  z <- (raf - nc_raf_median) / nc_raf_mad
  stats::pnorm(z / 3)
}

#' @noRd
raf_mp_concordance <- function(raf, mp, raf_threshold, mp_threshold) {
  if (!is.finite(raf) || !is.finite(mp)) return(0.5)
  raf_high <- raf > raf_threshold
  mp_high <- mp > mp_threshold
  if (raf_high && mp_high) return(1.0)
  if (!raf_high && mp_high) return(0.3)
  if (raf_high && !mp_high) return(0.5)
  return(0.0)
}

# --- TTT Scoring Functions (internal) ---

#' @noRd
ttt_binary <- function(ttt, threshold) {
  ifelse(!is.na(ttt) & ttt < threshold, 1, 0)
}

#' @noRd
ttt_sigmoid <- function(ttt, threshold, k = 2) {
  ifelse(is.finite(ttt), 1 / (1 + exp((ttt - threshold) / k)), 0)
}

#' @noRd
ttt_linear <- function(ttt, pc_median, nc_median) {
  den <- nc_median - pc_median
  if (!is.finite(den) || abs(den) < .Machine$double.eps) return(0.5)
  score <- (nc_median - ttt) / den
  pmin(1, pmax(0, score))
}

#' @noRd
ttt_percentile <- function(ttt, pc_ttt, nc_ttt) {
  if (!is.finite(ttt)) return(0)
  all_ttt <- c(pc_ttt, nc_ttt)
  ecdf_func <- stats::ecdf(all_ttt)
  1 - ecdf_func(ttt)
}

# --- Signal (MP) Scoring Functions (internal) ---

#' @noRd
signal_mpr <- function(mp, nc_median, cutoff = 2.0) {
  denom <- pmax(nc_median, .Machine$double.eps)
  mpr <- mp / denom
  ifelse(!is.na(mpr) & is.finite(mpr) & mpr >= cutoff, 1, 0)
}

#' @noRd
signal_mpr_continuous <- function(mp, nc_median, pc_median) {
  denom <- pc_median - nc_median
  if (!is.finite(denom) || abs(denom) < .Machine$double.eps) return(0.5)
  mpr <- (mp - nc_median) / denom
  pmin(1, pmax(0, mpr))
}

#' @noRd
signal_zscore_mad <- function(mp, nc_median, nc_mad) {
  z <- (mp - nc_median) / nc_mad
  stats::pnorm(z / 3)
}

#' @noRd
signal_percentile <- function(mp, pc_mp, nc_mp) {
  all_mp <- c(pc_mp, nc_mp)
  ecdf_func <- stats::ecdf(all_mp)
  ecdf_func(mp)
}


# =============================================================================
# MAIN ANALYSIS FUNCTION
# =============================================================================

#' KWELA Main Analysis Function
#'
#' Implements hierarchical adaptive classification for RT-QuIC data with
#' dual-mode operation and group-aware mixed-assay support.
#'
#' @description
#' Implements a 6-layer hierarchical adaptive classification system:
#' \describe{
#'   \item{Layer 1: Hard Gate}{Biological constraint filter with stochastic rescue (research mode only)}
#'   \item{Layer 2: Per-Well Scoring}{Profile-dependent adaptive transforms}
#'   \item{Layer 3: Adaptive Combination}{Separation-aware score combiner}
#'   \item{Layer 4: Adaptive Cutoff}{Youden-optimized threshold per plate/group}
#'   \item{Layer 5: Replicate Consensus}{Treatment-level classification}
#'   \item{Layer 6: Instability Detection}{Matrix interference override}
#' }
#'
#' @param data Data frame with Treatment, TTT, MP columns (RAF optional)
#' @param pc_pattern Regex for positive controls
#' @param nc_pattern Regex for negative controls
#' @param spiked_pattern Regex for spiked samples
#' @param profile Classification profile: "auto", "standard", "sensitive",
#'   or "matrix_robust". "auto" selects based on separation quality.
#' @param consensus Replicate consensus rule: "majority", "strict",
#'   "flexible", or "threshold"
#' @param consensus_threshold For "threshold" consensus: minimum mean
#'   well score to classify treatment as positive (default 0.5)
#' @param matrix_groups Optional: column name for matrix grouping
#'   (enables per-group PC/NC controls and thresholds). NULL = global.
#' @param use_raf Logical: include RAF in scoring (default TRUE if present)
#' @param mode "diagnostic" (deterministic, no stochastic rescue) or
#'   "research" (full adaptive architecture). Default: "diagnostic".
#' @param instability_check Logical: run instability detection (default TRUE)
#' @param instability_strictness "moderate" (2+ flags), "strict" (1+),
#'   or "lenient" (3+). Controls override sensitivity.
#' @param verbose Print progress
#' @return Data frame with per-well scores and classifications, plus attributes
#'   containing treatment-level consensus, separation metrics, group controls,
#'   mode, instability results, and thresholds
#' @export
#' @examples
#' \donttest{
#' set.seed(42)
#' df <- data.frame(
#'   Treatment = c(rep("Positive Control", 8), rep("Negative Control", 8),
#'                 rep("Sample_A", 8)),
#'   TTT = c(rnorm(8, 8, 1), rnorm(8, 72, 5), rnorm(8, 12, 3)),
#'   MP = c(rnorm(8, 100, 10), rnorm(8, 20, 5), rnorm(8, 85, 15))
#' )
#' result <- kwela_analyze(df)
#' summary <- kwela_summarize(result)
#' }
# =============================================================================
# KWELA 1.0.0 MAIN ANALYSIS - 6-LAYER ARCHITECTURE
# =============================================================================
kwela_analyze <- function(
    data,
    pc_pattern = "\\bPositive\\s*Control\\b|^POS\\b|\\bPC\\b",
    nc_pattern = "\\bNegative\\s*Control\\b|^NEG\\b|\\bNC\\b|\\bTDB\\b|\\bBlank\\b",
    spiked_pattern = "\\+\\s*Pos|Pos\\s*\\d*%|spiked|\\+\\s*CWD",
    profile = c("auto", "standard", "sensitive", "matrix_robust"),
    consensus = c("majority", "strict", "flexible", "threshold"),
    consensus_threshold = 0.5,
    matrix_groups = NULL,
    use_raf = TRUE,
    mode = c("diagnostic", "research"),
    instability_check = TRUE,
    instability_strictness = c("moderate", "strict", "lenient"),
    verbose = TRUE
) {
  profile <- match.arg(profile)
  consensus <- match.arg(consensus)
  mode <- match.arg(mode)
  instability_strictness <- match.arg(instability_strictness)

  if (verbose) {
    message("\n\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550")
    message("KWELA 1.0.0: Hierarchical Adaptive RT-QuIC Classification")
    message("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550")
    message(sprintf("  [MODE] %s | instability_check=%s (%s)",
                    toupper(mode), instability_check, instability_strictness))
  }

  # ── Data Preparation ──────────────────────────────────────────────────
  df <- as.data.frame(data)

  name_map <- c(
    "Treatment" = "Treatment|treatment|Sample\\.?IDs?|sample",
    "TTT" = "TTT|TtT|Time.to.Threshold|TimeToThreshold|time_to_threshold",
    "MS"  = "MS|Max.Slope|MaxSlope|Max_Slope|max_slope",
    "MP"  = "MP|MPR|Maxpoint|MaxPoint|Max_Point|Fmax|maxpoint",
    "RAF" = "RAF|Rate.of.Amyloid.Formation|rate_of_amyloid_formation"
  )

  for (std_name in names(name_map)) {
    pattern <- name_map[std_name]
    matches <- grep(pattern, names(df), ignore.case = TRUE, value = TRUE)
    if (length(matches) > 0 && !std_name %in% names(df)) {
      if (length(matches) > 1) {
        stop(sprintf(
          "Ambiguous mapping for '%s': matches (%s). Pre-standardize column names.",
          std_name, paste(matches, collapse = ", ")))
      }
      names(df)[names(df) == matches[1]] <- std_name
    }
  }

  for (col in c("TTT", "MS", "MP", "RAF")) {
    if (col %in% names(df)) df[[col]] <- suppressWarnings(as.numeric(df[[col]]))
  }

  has_raf <- "RAF" %in% names(df) && use_raf
  if (!has_raf && use_raf && verbose) {
    message("  [INFO] RAF column not found; RAF scoring disabled")
  }

  # ── Input Validation ──────────────────────────────────────────────────
  required_cols <- c("Treatment", "TTT", "MP")
  missing_cols <- setdiff(required_cols, names(df))
  if (length(missing_cols) > 0) {
    stop(sprintf("Missing required columns: %s", paste(missing_cols, collapse = ", ")))
  }
  if (nrow(df) == 0) stop("Input data has zero rows")

  # ── Identify Sample Types ─────────────────────────────────────────────
  df$Type <- "Sample"
  df$Type[grepl(pc_pattern, df$Treatment, ignore.case = TRUE, perl = TRUE)] <- "PC"
  df$Type[grepl(nc_pattern, df$Treatment, ignore.case = TRUE, perl = TRUE)] <- "NC"
  df$is_spiked <- grepl(spiked_pattern, df$Treatment, ignore.case = TRUE, perl = TRUE)

  pc_data <- df[df$Type == "PC", ]
  nc_data <- df[df$Type == "NC", ]

  if (nrow(pc_data) < 2) stop("Insufficient positive controls (need >= 2 wells)")
  if (nrow(nc_data) < 2) stop("Insufficient negative controls (need >= 2 wells)")

  if (verbose) {
    message(sprintf("\n  [Data] %d total wells: %d PC | %d NC | %d samples",
                    nrow(df), nrow(pc_data), nrow(nc_data), sum(df$Type == "Sample")))
  }

  # ── Extract Control Distributions ─────────────────────────────────────
  pc_ttt <- pc_data$TTT[is.finite(pc_data$TTT)]
  pc_mp  <- pc_data$MP[is.finite(pc_data$MP)]
  nc_ttt <- nc_data$TTT[is.finite(nc_data$TTT)]
  nc_mp  <- nc_data$MP[is.finite(nc_data$MP)]

  pc_stats <- list(
    ttt_mean = mean(pc_ttt, na.rm = TRUE), ttt_sd = safe_sd(pc_ttt),
    ttt_median = stats::median(pc_ttt, na.rm = TRUE), ttt_mad = robust_scale(pc_ttt),
    mp_mean = mean(pc_mp, na.rm = TRUE), mp_sd = safe_sd(pc_mp),
    mp_median = stats::median(pc_mp, na.rm = TRUE), mp_mad = robust_scale(pc_mp)
  )

  nc_stats <- list(
    ttt_mean = mean(nc_ttt, na.rm = TRUE), ttt_sd = safe_sd(nc_ttt),
    ttt_median = stats::median(nc_ttt, na.rm = TRUE), ttt_mad = robust_scale(nc_ttt),
    mp_mean = mean(nc_mp, na.rm = TRUE), mp_sd = safe_sd(nc_mp),
    mp_median = stats::median(nc_mp, na.rm = TRUE), mp_mad = robust_scale(nc_mp)
  )

  # RAF control stats — row-aligned to avoid length mismatch between
  # RAF and MP after independent is.finite() filtering
  if (has_raf) {
    # Compute ratios on row-aligned pairs (both RAF and MP finite for same well)
    pc_raf_mp_valid <- is.finite(pc_data$RAF) & is.finite(pc_data$MP)
    nc_raf_mp_valid <- is.finite(nc_data$RAF) & is.finite(nc_data$MP)
    pc_raf_mp_ratios <- pc_data$RAF[pc_raf_mp_valid] /
      pmax(pc_data$MP[pc_raf_mp_valid], .Machine$double.eps)
    nc_raf_mp_ratios <- nc_data$RAF[nc_raf_mp_valid] /
      pmax(nc_data$MP[nc_raf_mp_valid], .Machine$double.eps)

    nc_raf <- nc_data$RAF[is.finite(nc_data$RAF)]
    nc_raf_median <- stats::median(nc_raf, na.rm = TRUE)
    nc_raf_mad <- robust_scale(nc_raf)
    raf_threshold <- nc_raf_median + 1.5 * nc_raf_mad
  }

  # ── Control Sanity Checks ─────────────────────────────────────────────
  controls_inverted <- FALSE
  controls_degenerate <- FALSE

  if (length(pc_ttt) < 2 || length(nc_ttt) < 2) controls_degenerate <- TRUE

  if (!controls_degenerate) {
    pc_fast <- is.finite(pc_stats$ttt_median) && is.finite(nc_stats$ttt_median) &&
      (pc_stats$ttt_median < nc_stats$ttt_median)
    pc_high <- is.finite(pc_stats$mp_median) && is.finite(nc_stats$mp_median) &&
      (pc_stats$mp_median > nc_stats$mp_median)
    if (!pc_fast && !pc_high) controls_inverted <- TRUE

    if (!controls_inverted) {
      pc_nc_wass <- wasserstein_1d(pc_mp, nc_mp)
      nc_spread <- robust_scale(nc_mp)
      if (is.finite(pc_nc_wass) && is.finite(nc_spread) && nc_spread > 0) {
        if (pc_nc_wass / nc_spread < 0.5) {
          controls_degenerate <- TRUE
          if (verbose) message("  [WARN] PC/NC distributions overlap heavily (degenerate)")
        }
      }
    }
  }

  if (controls_degenerate || controls_inverted) {
    warning("Control sanity check failed: PC/NC appear degenerate or inverted.")
  }

  # ── Separation Quality Assessment (global baseline) ────────────────
  separation <- assess_separation(pc_mp, nc_mp, pc_ttt, nc_ttt)

  # NOTE: Auto-profile selection is now handled after group_controls are built
  # (group-aware separation → conservative profile hierarchy).
  # If no matrix_groups, falls back to global separation in that block.

  # ── Classification Thresholds (global defaults) ──────────────────────
  # NC TTT availability rate: fraction of NC wells with finite TTT values.
  # NOT a "crossing rate" — this measures data availability, not threshold crossing.
  # If <50% NC have TTT values, the midpoint threshold is unreliable.
  nc_ttt_avail_rate <- mean(is.finite(nc_data$TTT))

  if (nc_ttt_avail_rate >= 0.50 && length(nc_ttt) >= 2) {
    crossing_threshold <- (pc_stats$ttt_median + nc_stats$ttt_median) / 2
  } else {
    crossing_threshold <- as.numeric(stats::quantile(pc_ttt, 0.75, names = FALSE))
    if (verbose) message(sprintf("  [WARN] NC TTT availability %.0f%% - using PC Q75",
                                 nc_ttt_avail_rate * 100))
  }

  signal_threshold <- nc_stats$mp_median + 1.5 * nc_stats$mp_mad

  if (verbose) {
    message(sprintf("\n  [Thresholds] TTT < %.1f | MP > %.1f",
                    crossing_threshold, signal_threshold))
    message(sprintf("  [Controls] PC: TTT=%.1f, MP=%.1f | NC: TTT=%.1f, MP=%.1f",
                    pc_stats$ttt_median, pc_stats$mp_median,
                    nc_stats$ttt_median, nc_stats$mp_median))
  }

  # ── Group-Aware Controls (assay/plate/matrix) ─────────────────────────
  # If matrix_groups is set (e.g., Assay = "RT"/"Nano", or Plate), compute
  # per-group PC/NC distributions, stats, thresholds, and RAF params.
  # This makes mixed RT-QuIC + Nano-QuIC plates behave like two valid assays
  # analyzed under one call. Falls back to global controls if insufficient.
  group_controls <- NULL

  if (!is.null(matrix_groups) && matrix_groups %in% names(df)) {
    if (verbose) message("\n  [MATRIX] Computing per-group PC/NC controls + thresholds...")

    groups <- unique(df[[matrix_groups]][df$Type %in% c("Sample", "PC", "NC")])
    groups <- groups[!is.na(groups)]
    group_controls <- list()

    for (grp in groups) {
      grp_df <- df[df[[matrix_groups]] == grp, ]
      grp_pc <- grp_df[grp_df$Type == "PC", ]
      grp_nc <- grp_df[grp_df$Type == "NC", ]

      # Extract group control vectors
      grp_pc_ttt <- grp_pc$TTT[is.finite(grp_pc$TTT)]
      grp_pc_mp  <- grp_pc$MP[is.finite(grp_pc$MP)]
      grp_nc_ttt <- grp_nc$TTT[is.finite(grp_nc$TTT)]
      grp_nc_mp  <- grp_nc$MP[is.finite(grp_nc$MP)]

      ok_pc <- length(grp_pc_mp) >= 2
      ok_nc <- length(grp_nc_mp) >= 2

      if (!(ok_pc && ok_nc)) {
        # Fallback to global controls
        group_controls[[as.character(grp)]] <- list(
          pc_ttt = pc_ttt, pc_mp = pc_mp, nc_ttt = nc_ttt, nc_mp = nc_mp,
          pc_stats = pc_stats, nc_stats = nc_stats,
          crossing_threshold = crossing_threshold,
          signal_threshold = signal_threshold,
          has_raf = has_raf,
          pc_raf_mp_ratios = if (has_raf) pc_raf_mp_ratios else NULL,
          nc_raf_mp_ratios = if (has_raf) nc_raf_mp_ratios else NULL,
          nc_raf_median = if (has_raf) nc_raf_median else NA_real_,
          nc_raf_mad = if (has_raf) nc_raf_mad else NA_real_,
          raf_threshold = if (has_raf) raf_threshold else NA_real_
        )
        if (verbose) message(sprintf("    Group '%s': insufficient local controls -> using global", grp))
        next
      }

      grp_pc_stats <- list(
        ttt_mean = mean(grp_pc_ttt, na.rm = TRUE), ttt_sd = safe_sd(grp_pc_ttt),
        ttt_median = stats::median(grp_pc_ttt, na.rm = TRUE), ttt_mad = robust_scale(grp_pc_ttt),
        mp_mean = mean(grp_pc_mp, na.rm = TRUE), mp_sd = safe_sd(grp_pc_mp),
        mp_median = stats::median(grp_pc_mp, na.rm = TRUE), mp_mad = robust_scale(grp_pc_mp)
      )

      grp_nc_stats <- list(
        ttt_mean = mean(grp_nc_ttt, na.rm = TRUE), ttt_sd = safe_sd(grp_nc_ttt),
        ttt_median = stats::median(grp_nc_ttt, na.rm = TRUE), ttt_mad = robust_scale(grp_nc_ttt),
        mp_mean = mean(grp_nc_mp, na.rm = TRUE), mp_sd = safe_sd(grp_nc_mp),
        mp_median = stats::median(grp_nc_mp, na.rm = TRUE), mp_mad = robust_scale(grp_nc_mp)
      )

      # Group-specific thresholds
      grp_nc_ttt_avail <- mean(is.finite(grp_nc$TTT))
      grp_crossing_threshold <- if (grp_nc_ttt_avail >= 0.50 && length(grp_nc_ttt) >= 2 && length(grp_pc_ttt) >= 2) {
        (grp_pc_stats$ttt_median + grp_nc_stats$ttt_median) / 2
      } else if (length(grp_pc_ttt) >= 2) {
        as.numeric(stats::quantile(grp_pc_ttt, 0.75, names = FALSE))
      } else {
        crossing_threshold
      }

      grp_signal_threshold <- grp_nc_stats$mp_median + 1.5 * grp_nc_stats$mp_mad

      # RAF per-group params
      grp_pc_raf_mp_ratios <- NULL
      grp_nc_raf_mp_ratios <- NULL
      grp_nc_raf_median <- NA_real_
      grp_nc_raf_mad <- NA_real_
      grp_raf_threshold <- NA_real_

      if (has_raf) {
        pc_ok <- is.finite(grp_pc$RAF) & is.finite(grp_pc$MP)
        nc_ok <- is.finite(grp_nc$RAF) & is.finite(grp_nc$MP)
        grp_pc_raf_mp_ratios <- grp_pc$RAF[pc_ok] / pmax(grp_pc$MP[pc_ok], .Machine$double.eps)
        grp_nc_raf_mp_ratios <- grp_nc$RAF[nc_ok] / pmax(grp_nc$MP[nc_ok], .Machine$double.eps)

        grp_nc_raf <- grp_nc$RAF[is.finite(grp_nc$RAF)]
        grp_nc_raf_median <- stats::median(grp_nc_raf, na.rm = TRUE)
        grp_nc_raf_mad <- robust_scale(grp_nc_raf)
        grp_raf_threshold <- grp_nc_raf_median + 1.5 * grp_nc_raf_mad
      }

      group_controls[[as.character(grp)]] <- list(
        pc_ttt = grp_pc_ttt, pc_mp = grp_pc_mp, nc_ttt = grp_nc_ttt, nc_mp = grp_nc_mp,
        pc_stats = grp_pc_stats, nc_stats = grp_nc_stats,
        crossing_threshold = grp_crossing_threshold,
        signal_threshold = grp_signal_threshold,
        has_raf = has_raf,
        pc_raf_mp_ratios = grp_pc_raf_mp_ratios,
        nc_raf_mp_ratios = grp_nc_raf_mp_ratios,
        nc_raf_median = grp_nc_raf_median,
        nc_raf_mad = grp_nc_raf_mad,
        raf_threshold = grp_raf_threshold
      )

      if (verbose) {
        message(sprintf(
          "    Group '%s': PC=%d, NC=%d | TTT<%.1f | MP>%.1f",
          grp, nrow(grp_pc), nrow(grp_nc), grp_crossing_threshold, grp_signal_threshold
        ))
      }
    }
  } else if (!is.null(matrix_groups) && !matrix_groups %in% names(df)) {
    warning(sprintf("matrix_groups column '%s' not found in data. Using global controls.", matrix_groups))
  }

  # ── Per-group separation assessment ──────────────────────────────────
  group_separation <- NULL
  group_profiles <- NULL

  if (!is.null(group_controls)) {
    group_separation <- list()
    group_profiles <- list()

    for (grp in names(group_controls)) {
      gc <- group_controls[[grp]]
      sep <- assess_separation(
        pc_mp = gc$pc_mp, nc_mp = gc$nc_mp,
        pc_ttt = gc$pc_ttt, nc_ttt = gc$nc_ttt
      )
      group_separation[[grp]] <- sep
      group_profiles[[grp]] <- sep$recommended_profile
    }
  }

  # ── Auto-profile selection (group-aware) ─────────────────────────────
  if (profile == "auto") {
    if (!is.null(group_profiles) && length(group_profiles) > 0) {
      # Conservative hierarchy: matrix_robust > sensitive > standard
      # If ANY group is messy, use the safer profile for the whole run.
      prof_vec <- unlist(group_profiles, use.names = FALSE)
      if ("matrix_robust" %in% prof_vec) {
        profile <- "matrix_robust"
      } else if ("sensitive" %in% prof_vec) {
        profile <- "sensitive"
      } else {
        profile <- "standard"
      }
      if (verbose) {
        message("\n  [AUTO] Per-group separation:")
        for (grp in names(group_profiles)) {
          sep <- group_separation[[grp]]
          message(sprintf("    %s -> %s (d = %.2f)",
                          grp, toupper(group_profiles[[grp]]), sep$d_combined))
        }
        message(sprintf("  [AUTO] Selected global profile: %s", toupper(profile)))
      }
    } else {
      profile <- separation$recommended_profile
      if (verbose) {
        message(sprintf("\n  [AUTO] Separation regime: %s (Cohen's d = %.2f)",
                        separation$regime, separation$d_combined))
        message(sprintf("  [AUTO] Selected profile: %s", toupper(profile)))
      }
    }
  } else if (verbose) {
    message(sprintf("\n  [PROFILE] Using user-specified profile: %s", toupper(profile)))
    message(sprintf("  [INFO] Separation: %s (d = %.2f)",
                    separation$regime, separation$d_combined))
  }

  # ══════════════════════════════════════════════════════════════════════
  # PRE-COMPUTE: Treatment-level stochastic metrics
  # (Needed before hard gate for stochastic rescue, and in Layer 2)
  # ══════════════════════════════════════════════════════════════════════
  df$ssmd_vs_nc <- NA_real_
  df$wasserstein_vs_pc <- NA_real_
  df$wasserstein_vs_nc <- NA_real_
  df$energy_vs_pc <- NA_real_
  df$energy_vs_nc <- NA_real_

  trt_stoch <- list()
  treatments <- unique(df$Treatment[df$Type == "Sample"])

  for (trt in treatments) {
    idx <- which(df$Treatment == trt & df$Type == "Sample")
    trt_mp <- df$MP[idx]
    trt_mp_finite <- trt_mp[is.finite(trt_mp)]

    if (length(trt_mp_finite) >= 2) {
      wass_pc <- wasserstein_1d(trt_mp_finite, pc_mp)
      wass_nc <- wasserstein_1d(trt_mp_finite, nc_mp)
      energy_pc <- energy_distance(trt_mp_finite, pc_mp)
      energy_nc <- energy_distance(trt_mp_finite, nc_mp)
      ssmd_nc <- ssmd(trt_mp_finite, nc_mp)

      if (is.finite(wass_pc)) wass_pc <- max(wass_pc, 0)
      if (is.finite(wass_nc)) wass_nc <- max(wass_nc, 0)
      if (is.finite(energy_pc)) energy_pc <- max(energy_pc, 0)
      if (is.finite(energy_nc)) energy_nc <- max(energy_nc, 0)

      wass_score <- safe_ratio(wass_pc, wass_nc)
      # SSMD score: map to [0,1] via pnorm with scaling constant 2.5.
      # Rationale: SSMD of ~2.5 (strong effect) maps to ~0.84 (1 SD above mean);
      # SSMD of ~5.0 maps to ~0.98. The constant compresses the dynamic range
      # so moderate effects (~1-3 SSMD) produce meaningful score differentiation.
      # This is a heuristic mapping, not a p-value. Clamped to [-8,8] pre-scaling
      # to prevent numeric overflow in pnorm.
      ssmd_score <- if (!is.na(ssmd_nc)) stats::pnorm(pmin(8, pmax(-8, ssmd_nc / 2.5))) else 0.5

      trt_stoch[[trt]] <- list(
        wass_score = wass_score %||% 0.5,
        ssmd_score = ssmd_score,
        wass_pc = wass_pc, wass_nc = wass_nc,
        energy_pc = energy_pc, energy_nc = energy_nc,
        ssmd_nc = ssmd_nc
      )

      df$ssmd_vs_nc[idx] <- ssmd_nc
      df$wasserstein_vs_pc[idx] <- wass_pc
      df$wasserstein_vs_nc[idx] <- wass_nc
      df$energy_vs_pc[idx] <- energy_pc
      df$energy_vs_nc[idx] <- energy_nc
    }
  }

  # ══════════════════════════════════════════════════════════════════════
  # LAYER 1: HARD GATE (biological constraint filter)
  # ══════════════════════════════════════════════════════════════════════
  if (verbose) message("\n  [L1] Hard Gate: biological constraint filter...")

  # Group-aware gate thresholds (per well)
  df$crossed_threshold <- FALSE
  df$has_signal <- FALSE

  for (i in seq_len(nrow(df))) {
    if (df$Type[i] %in% c("PC", "NC")) next

    grp <- if (!is.null(group_controls) && !is.null(matrix_groups)) df[[matrix_groups]][i] else NA
    gc <- if (!is.na(grp) && !is.null(group_controls[[as.character(grp)]])) group_controls[[as.character(grp)]] else NULL

    ct <- if (!is.null(gc)) gc$crossing_threshold else crossing_threshold
    st <- if (!is.null(gc)) gc$signal_threshold else signal_threshold

    df$crossed_threshold[i] <- is.finite(df$TTT[i]) && df$TTT[i] < ct
    df$has_signal[i] <- is.finite(df$MP[i]) && df$MP[i] > st
  }
  df$hard_gate_pass <- FALSE

  # Hard gate: must have EITHER crossed threshold OR signal above NC
  # RAF artifact filter: if MP is high but RAF/MP ratio is abnormally low,
  # flag as artifact (MP high but no kinetic signature)
  df$artifact_flag <- FALSE
  df$stoch_rescue <- FALSE

  # Pre-compute treatment-level rescue eligibility (not per-well).
  # DIAGNOSTIC MODE: rescue disabled — classification based on individual evidence only.
  # RESEARCH MODE: rescue enabled for treatments with distributional SSMD >= 0.7.
  rescue_eligible_trts <- character(0)
  if (mode == "research") {
    for (trt in treatments) {
      trt_n_finite <- sum(is.finite(df$MP[df$Treatment == trt & df$Type == "Sample"]))
      if (trt_n_finite < 3) next  # Insufficient replicates for stable SSMD
      if (!is.null(trt_stoch[[trt]])) {
        stoch <- trt_stoch[[trt]]
        if (is.finite(stoch$ssmd_score) && stoch$ssmd_score >= 0.7) {
          rescue_eligible_trts <- c(rescue_eligible_trts, trt)
        }
      }
    }
  }

  for (i in seq_len(nrow(df))) {
    if (df$Type[i] %in% c("PC", "NC")) next

    crossed <- df$crossed_threshold[i]
    has_sig <- df$has_signal[i]

    # Basic gate: need some evidence of positivity
    # Stochastic rescue: if neither threshold crossed nor signal present,
    # but treatment is rescue-eligible (distributional evidence of positivity),
    # allow through with confidence downgrade (prevents losing late/weak seeders)
    if (!crossed && !has_sig) {
      trt <- df$Treatment[i]
      if (trt %in% rescue_eligible_trts) {
        df$hard_gate_pass[i] <- TRUE
        df$stoch_rescue[i] <- TRUE
        next
      }
      df$hard_gate_pass[i] <- FALSE
      next
    }

    # RAF artifact check: high MP without kinetic signature
    if (has_raf && has_sig && !crossed) {
      raf_val <- df$RAF[i]
      mp_val <- df$MP[i]
      if (is.finite(raf_val) && is.finite(mp_val) && mp_val > .Machine$double.eps) {
        ratio <- raf_val / mp_val
        # If RAF/MP ratio is below NC 25th percentile ratio, likely artifact
        # Using percentile (not median) for robustness across matrix types
        # Use group-local NC ratios when available
        grp <- if (!is.null(group_controls) && !is.null(matrix_groups)) df[[matrix_groups]][i] else NA
        gc <- if (!is.na(grp) && !is.null(group_controls[[as.character(grp)]])) group_controls[[as.character(grp)]] else NULL
        nc_ratios <- if (!is.null(gc) && !is.null(gc$nc_raf_mp_ratios)) gc$nc_raf_mp_ratios else nc_raf_mp_ratios
        nc_ratios_finite <- nc_ratios[is.finite(nc_ratios)]
        nc_ratio_q25 <- if (length(nc_ratios_finite) >= 3) {
          as.numeric(stats::quantile(nc_ratios_finite, 0.25, names = FALSE))
        } else {
          stats::median(nc_ratios_finite, na.rm = TRUE)
        }
        if (is.finite(nc_ratio_q25) && ratio < nc_ratio_q25) {
          df$artifact_flag[i] <- TRUE
          df$hard_gate_pass[i] <- FALSE
          next
        }
      }
    }

    df$hard_gate_pass[i] <- TRUE
  }

  n_gated <- sum(!df$hard_gate_pass & df$Type == "Sample")
  n_rescued <- sum(df$stoch_rescue & df$Type == "Sample")
  if (verbose) {
    message(sprintf("    %d/%d sample wells gated out (negative or artifact)",
                               n_gated, sum(df$Type == "Sample")))
    if (n_rescued > 0) {
      message(sprintf("    %d wells rescued via treatment-level stochastic evidence (%d treatments eligible)",
                      n_rescued, length(rescue_eligible_trts)))
    }
  }

  # ══════════════════════════════════════════════════════════════════════
  # LAYER 2: PER-WELL SCORING (profile-dependent)
  # ══════════════════════════════════════════════════════════════════════
  if (verbose) message("\n  [L2] Per-well scoring...")

  df$ttt_score <- NA_real_
  df$signal_score <- NA_real_
  df$raf_score <- NA_real_
  df$stoch_score <- NA_real_
  df$well_score <- NA_real_

  # trt_stoch and distributional metrics already computed above (pre-gate)

  # Score each well
  sample_idx <- which(df$Type == "Sample")

  for (i in sample_idx) {
    # Wells that failed hard gate get score 0
    if (!df$hard_gate_pass[i]) {
      df$ttt_score[i] <- 0
      df$signal_score[i] <- 0
      df$raf_score[i] <- 0
      df$stoch_score[i] <- 0
      df$well_score[i] <- 0
      next
    }

    ttt_val <- df$TTT[i]
    mp_val <- df$MP[i]
    trt <- df$Treatment[i]

    # Resolve group-local vs global controls for this well
    grp <- if (!is.null(group_controls) && !is.null(matrix_groups)) df[[matrix_groups]][i] else NA
    gc <- if (!is.na(grp) && !is.null(group_controls[[as.character(grp)]])) group_controls[[as.character(grp)]] else NULL

    pc_ttt_loc <- if (!is.null(gc)) gc$pc_ttt else pc_ttt
    nc_ttt_loc <- if (!is.null(gc)) gc$nc_ttt else nc_ttt
    pc_stats_loc <- if (!is.null(gc)) gc$pc_stats else pc_stats
    nc_stats_loc <- if (!is.null(gc)) gc$nc_stats else nc_stats
    crossing_threshold_loc <- if (!is.null(gc)) gc$crossing_threshold else crossing_threshold
    signal_threshold_loc <- if (!is.null(gc)) gc$signal_threshold else signal_threshold

    local_nc_mp_med <- nc_stats_loc$mp_median
    local_nc_mp_mad <- nc_stats_loc$mp_mad
    local_nc_ttt_med <- nc_stats_loc$ttt_median

    # --- TTT Score ---
    if (profile == "standard") {
      df$ttt_score[i] <- if (is.finite(ttt_val)) {
        ttt_percentile(ttt_val, pc_ttt_loc, nc_ttt_loc)
      } else 0
    } else if (profile == "sensitive") {
      df$ttt_score[i] <- if (is.finite(ttt_val)) {
        ttt_sigmoid(ttt_val, crossing_threshold_loc, k = 2)
      } else 0
    } else {
      # matrix_robust: linear with local TTT baseline
      df$ttt_score[i] <- if (is.finite(ttt_val)) {
        ttt_linear(ttt_val, pc_stats_loc$ttt_median, local_nc_ttt_med)
      } else 0
    }

    # --- Signal (MP) Score --- (group-local baselines)
    if (profile == "standard") {
      df$signal_score[i] <- if (is.finite(mp_val)) {
        signal_mpr(mp_val, local_nc_mp_med, cutoff = 2.0)
      } else 0
    } else if (profile == "sensitive") {
      df$signal_score[i] <- if (is.finite(mp_val)) {
        signal_zscore_mad(mp_val, local_nc_mp_med, local_nc_mp_mad)
      } else 0
    } else {
      # matrix_robust: continuous MPR with group-local baselines
      df$signal_score[i] <- if (is.finite(mp_val)) {
        signal_mpr_continuous(mp_val, local_nc_mp_med, pc_stats_loc$mp_median)
      } else 0
    }

    # --- RAF Score --- (group-local RAF params)
    if (has_raf) {
      raf_val <- df$RAF[i]
      pc_rat <- if (!is.null(gc) && !is.null(gc$pc_raf_mp_ratios)) gc$pc_raf_mp_ratios else pc_raf_mp_ratios
      nc_rat <- if (!is.null(gc) && !is.null(gc$nc_raf_mp_ratios)) gc$nc_raf_mp_ratios else nc_raf_mp_ratios
      nc_raf_med_loc <- if (!is.null(gc)) gc$nc_raf_median else nc_raf_median
      nc_raf_mad_loc <- if (!is.null(gc)) gc$nc_raf_mad else nc_raf_mad
      raf_thresh_loc <- if (!is.null(gc)) gc$raf_threshold else raf_threshold

      if (profile == "matrix_robust") {
        df$raf_score[i] <- raf_ratio_score(raf_val, mp_val, pc_rat, nc_rat)
      } else if (profile == "sensitive") {
        df$raf_score[i] <- raf_mp_concordance(raf_val, mp_val,
                                               raf_thresh_loc, signal_threshold_loc)
      } else {
        df$raf_score[i] <- raf_zscore(raf_val, nc_raf_med_loc, nc_raf_mad_loc)
      }
    } else {
      df$raf_score[i] <- NA_real_  # Will be excluded from combination
    }

    # --- Stochastic Score (treatment-level, assigned to wells) ---
    if (!is.null(trt_stoch[[trt]])) {
      stoch <- trt_stoch[[trt]]
      df$stoch_score[i] <- (stoch$wass_score + stoch$ssmd_score) / 2
    } else {
      df$stoch_score[i] <- 0.5
    }
  }

  # ══════════════════════════════════════════════════════════════════════
  # LAYER 3: ADAPTIVE COMBINATION
  # ══════════════════════════════════════════════════════════════════════
  if (verbose) message("  [L3] Adaptive combination...")

  for (i in sample_idx) {
    if (!df$hard_gate_pass[i]) next  # Already scored 0

    t_sc <- df$ttt_score[i]
    s_sc <- df$signal_score[i]
    r_sc <- df$raf_score[i]
    st_sc <- df$stoch_score[i]

    if (profile == "standard") {
      # Clean mode: max of available scores (permutation-validated)
      scores <- c(t_sc, s_sc)
      if (has_raf && is.finite(r_sc)) scores <- c(scores, r_sc)
      df$well_score[i] <- max(scores, na.rm = TRUE)

    } else if (profile == "sensitive") {
      # Robust mode: min gate (permutation-validated)
      # DIAGNOSTIC: exclude stochastic score — classification from TTT/MP/RAF only
      # RESEARCH: include stochastic score in min gate
      scores <- c(t_sc, s_sc)
      if (mode == "research") scores <- c(scores, st_sc)
      if (has_raf && is.finite(r_sc)) scores <- c(scores, r_sc)
      scores <- scores[is.finite(scores)]
      df$well_score[i] <- if (length(scores) > 0) min(scores) else 0

    } else {
      # Matrix-robust: min gate, high conservatism, no stochastic inflation
      scores <- c(s_sc)  # Signal-dominant (TTT weight = 0 per permutation)
      if (has_raf && is.finite(r_sc)) scores <- c(scores, r_sc)
      scores <- scores[is.finite(scores)]
      df$well_score[i] <- if (length(scores) > 0) min(scores) else 0
    }
  }

  # ══════════════════════════════════════════════════════════════════════
  # LAYER 4: ADAPTIVE CUTOFF (Youden-optimized on controls)
  # ══════════════════════════════════════════════════════════════════════
  if (verbose) message("  [L4] Adaptive cutoff (Youden optimization)...")

  # Compute well scores for controls to find optimal cutoff
  # Uses group-local distributions when matrix_groups is set
  pc_scores <- numeric(nrow(pc_data))
  nc_scores <- numeric(nrow(nc_data))

  for (j in seq_len(nrow(pc_data))) {
    ttt_val <- pc_data$TTT[j]
    mp_val <- pc_data$MP[j]

    # Resolve group-local controls for this PC well
    grp <- if (!is.null(group_controls) && !is.null(matrix_groups) && matrix_groups %in% names(pc_data)) pc_data[[matrix_groups]][j] else NA
    gc <- if (!is.na(grp) && !is.null(group_controls[[as.character(grp)]])) group_controls[[as.character(grp)]] else NULL
    pc_ttt_loc <- if (!is.null(gc)) gc$pc_ttt else pc_ttt
    nc_ttt_loc <- if (!is.null(gc)) gc$nc_ttt else nc_ttt
    pc_stats_loc <- if (!is.null(gc)) gc$pc_stats else pc_stats
    nc_stats_loc <- if (!is.null(gc)) gc$nc_stats else nc_stats
    crossing_threshold_loc <- if (!is.null(gc)) gc$crossing_threshold else crossing_threshold

    if (profile == "standard") {
      t_s <- if (is.finite(ttt_val)) ttt_percentile(ttt_val, pc_ttt_loc, nc_ttt_loc) else 0
      s_s <- if (is.finite(mp_val)) signal_mpr(mp_val, nc_stats_loc$mp_median, 2.0) else 0
      pc_scores[j] <- max(t_s, s_s)
    } else if (profile == "sensitive") {
      t_s <- if (is.finite(ttt_val)) ttt_sigmoid(ttt_val, crossing_threshold_loc, 2) else 0
      s_s <- if (is.finite(mp_val)) signal_zscore_mad(mp_val, nc_stats_loc$mp_median, nc_stats_loc$mp_mad) else 0
      pc_scores[j] <- min(t_s, s_s)
    } else {
      s_s <- if (is.finite(mp_val)) signal_mpr_continuous(mp_val, nc_stats_loc$mp_median, pc_stats_loc$mp_median) else 0
      pc_scores[j] <- s_s
    }
  }

  for (j in seq_len(nrow(nc_data))) {
    ttt_val <- nc_data$TTT[j]
    mp_val <- nc_data$MP[j]

    # Resolve group-local controls for this NC well
    grp <- if (!is.null(group_controls) && !is.null(matrix_groups) && matrix_groups %in% names(nc_data)) nc_data[[matrix_groups]][j] else NA
    gc <- if (!is.na(grp) && !is.null(group_controls[[as.character(grp)]])) group_controls[[as.character(grp)]] else NULL
    pc_ttt_loc <- if (!is.null(gc)) gc$pc_ttt else pc_ttt
    nc_ttt_loc <- if (!is.null(gc)) gc$nc_ttt else nc_ttt
    pc_stats_loc <- if (!is.null(gc)) gc$pc_stats else pc_stats
    nc_stats_loc <- if (!is.null(gc)) gc$nc_stats else nc_stats
    crossing_threshold_loc <- if (!is.null(gc)) gc$crossing_threshold else crossing_threshold

    if (profile == "standard") {
      t_s <- if (is.finite(ttt_val)) ttt_percentile(ttt_val, pc_ttt_loc, nc_ttt_loc) else 0
      s_s <- if (is.finite(mp_val)) signal_mpr(mp_val, nc_stats_loc$mp_median, 2.0) else 0
      nc_scores[j] <- max(t_s, s_s)
    } else if (profile == "sensitive") {
      t_s <- if (is.finite(ttt_val)) ttt_sigmoid(ttt_val, crossing_threshold_loc, 2) else 0
      s_s <- if (is.finite(mp_val)) signal_zscore_mad(mp_val, nc_stats_loc$mp_median, nc_stats_loc$mp_mad) else 0
      nc_scores[j] <- min(t_s, s_s)
    } else {
      s_s <- if (is.finite(mp_val)) signal_mpr_continuous(mp_val, nc_stats_loc$mp_median, pc_stats_loc$mp_median) else 0
      nc_scores[j] <- s_s
    }
  }

  # Youden optimization: find cutoff maximizing Sens + Spec - 1
  # Prefer spiked/unspiked well scores when available (more representative
  # of actual detection performance than PC/NC which are extremes).
  # Fall back to PC/NC scores when no spiked data exists.
  spiked_well_scores <- df$well_score[df$Type == "Sample" & df$is_spiked]
  unspiked_well_scores <- df$well_score[df$Type == "Sample" & !df$is_spiked]

  if (length(spiked_well_scores) >= 5 && length(unspiked_well_scores) >= 5) {
    youden_pos <- spiked_well_scores[is.finite(spiked_well_scores)]
    youden_neg <- unspiked_well_scores[is.finite(unspiked_well_scores)]
    youden_source <- "spiked/unspiked"
  } else {
    youden_pos <- pc_scores
    youden_neg <- nc_scores
    youden_source <- "PC/NC"
  }

  # Use actual observed scores + midpoints as candidates (exact optimization)
  u <- sort(unique(c(youden_pos, youden_neg)))
  u <- u[is.finite(u)]
  mids <- if (length(u) > 1) (u[-1] + u[-length(u)]) / 2 else numeric(0)
  candidate_cutoffs <- unique(sort(c(u, mids, seq(0.05, 0.95, by = 0.05))))
  candidate_cutoffs <- candidate_cutoffs[candidate_cutoffs > 0 &
                                          candidate_cutoffs < 1]
  best_youden <- -Inf
  optimal_cutoff <- 0.50

  for (cut in candidate_cutoffs) {
    sens <- mean(youden_pos >= cut)
    spec <- mean(youden_neg < cut)
    youden <- sens + spec - 1
    if (youden > best_youden) {
      best_youden <- youden
      optimal_cutoff <- cut
    }
  }

  if (verbose) {
    message(sprintf("    Optimal cutoff: %.2f (Youden J = %.3f, source: %s)",
                    optimal_cutoff, best_youden, youden_source))
  }

  # Classify wells
  df$well_positive <- FALSE
  df$well_positive[sample_idx] <- df$well_score[sample_idx] >= optimal_cutoff

  # Confidence assignment
  df$seed_score <- df$well_score
  df$classification <- NA_character_
  df$confidence <- NA_character_

  for (i in sample_idx) {
    if (controls_degenerate || controls_inverted) {
      df$classification[i] <- "INVALID"
      df$confidence[i] <- "CONTROL_CHECK_FAIL"
      next
    }

    P <- df$well_score[i]
    if (is.na(P)) {
      df$classification[i] <- "INVALID"
      df$confidence[i] <- "N/A"
      next
    }

    if (df$artifact_flag[i]) {
      df$classification[i] <- "ARTIFACT_SUSPECT"
      df$confidence[i] <- "LOW"
    } else if (df$well_positive[i]) {
      margin <- P - optimal_cutoff
      if (margin > 0.25) {
        df$classification[i] <- "POSITIVE"
        df$confidence[i] <- "VERY_HIGH"
      } else if (margin > 0.10) {
        df$classification[i] <- "POSITIVE"
        df$confidence[i] <- "HIGH"
      } else {
        df$classification[i] <- "POSITIVE"
        df$confidence[i] <- "MODERATE"
      }
    } else {
      deficit <- optimal_cutoff - P
      if (deficit > 0.25) {
        df$classification[i] <- "NEGATIVE"
        df$confidence[i] <- "HIGH"
      } else if (deficit > 0.10) {
        df$classification[i] <- "NEGATIVE"
        df$confidence[i] <- "MODERATE"
      } else {
        df$classification[i] <- "INCONCLUSIVE"
        df$confidence[i] <- "LOW"
      }
    }

    # Stochastic rescue confidence penalty: wells rescued by treatment-level
    # stochastic evidence (no individual threshold/signal) get capped at LOW
    if (isTRUE(df$stoch_rescue[i]) && df$classification[i] == "POSITIVE") {
      df$confidence[i] <- "LOW"
    }
  }

  # Mark controls
  df$classification[df$Type == "PC"] <- "PC"
  df$confidence[df$Type == "PC"] <- "CONTROL"
  df$classification[df$Type == "NC"] <- "NC"
  df$confidence[df$Type == "NC"] <- "CONTROL"

  # ══════════════════════════════════════════════════════════════════════
  # LAYER 5: REPLICATE CONSENSUS
  # ══════════════════════════════════════════════════════════════════════
  if (verbose) message("\n  [L5] Replicate consensus...")

  df$trt_classification <- NA_character_
  df$trt_confidence <- NA_character_
  df$trt_positive_rate <- NA_real_
  df$trt_mean_score <- NA_real_
  df$trt_n_wells <- NA_integer_
  df$trt_n_positive <- NA_integer_

  trt_summaries <- list()

  for (trt in treatments) {
    idx <- which(df$Treatment == trt & df$Type == "Sample")
    n <- length(idx)
    if (n == 0) next

    well_pos <- df$well_positive[idx]
    well_scores <- df$well_score[idx]
    n_pos <- sum(well_pos, na.rm = TRUE)
    pos_rate <- n_pos / n
    mean_score <- mean(well_scores, na.rm = TRUE)

    # Apply consensus rule
    trt_positive <- switch(consensus,
      "strict"    = all(well_pos, na.rm = TRUE),
      "majority"  = pos_rate > 0.5,
      "flexible"  = any(well_pos, na.rm = TRUE),
      "threshold" = mean_score >= consensus_threshold,
      pos_rate > 0.5  # default: majority
    )

    # Confidence from agreement level
    if (trt_positive) {
      if (pos_rate >= 0.75) {
        trt_conf <- "HIGH"
      } else if (pos_rate >= 0.50) {
        trt_conf <- "MODERATE"
      } else {
        trt_conf <- "LOW"
      }
      trt_class <- "POSITIVE"
    } else {
      if (pos_rate <= 0.0) {
        trt_conf <- "HIGH"
      } else if (pos_rate <= 0.25) {
        trt_conf <- "MODERATE"
      } else {
        trt_conf <- "LOW"
      }
      trt_class <- if (pos_rate > 0 && pos_rate <= 0.5) "INCONCLUSIVE" else "NEGATIVE"
    }

    df$trt_classification[idx] <- trt_class
    df$trt_confidence[idx] <- trt_conf
    df$trt_positive_rate[idx] <- pos_rate
    df$trt_mean_score[idx] <- mean_score
    df$trt_n_wells[idx] <- as.integer(n)
    df$trt_n_positive[idx] <- as.integer(n_pos)

    trt_summaries[[trt]] <- data.frame(
      Treatment = trt,
      n_wells = n,
      n_positive = n_pos,
      positive_rate = round(pos_rate, 3),
      mean_score = round(mean_score, 3),
      classification = trt_class,
      confidence = trt_conf,
      consensus_rule = consensus,
      is_spiked = any(df$is_spiked[idx]),
      stringsAsFactors = FALSE
    )
  }

  trt_summary_df <- do.call(rbind, trt_summaries)
  if (!is.null(trt_summary_df)) {
    trt_summary_df <- trt_summary_df[order(-trt_summary_df$positive_rate,
                                            -trt_summary_df$mean_score), ]
  }

  # ======================================================================
  # LAYER 6: INSTABILITY DETECTION (post-consensus override)
  # ======================================================================
  df$matrix_instability <- FALSE
  instability_summary <- list()

  if (instability_check && length(treatments) > 0) {
    if (verbose) message("\n  [L6] Instability detection...")

    for (trt in treatments) {
      idx <- which(df$Treatment == trt & df$Type == "Sample")
      if (length(idx) < 2) next

      trt_mp_vals <- df$MP[idx]
      trt_mp_fin <- trt_mp_vals[is.finite(trt_mp_vals)]
      if (length(trt_mp_fin) < 2) next

      trt_ttt_vals <- df$TTT[idx]

      # Resolve group-local controls for instability check
      grp <- if (!is.null(group_controls) && !is.null(matrix_groups)) {
        ug <- unique(df[[matrix_groups]][idx])
        if (length(ug) == 1 && !is.na(ug)) ug else NA
      } else NA
      gc_loc <- if (!is.na(grp) && !is.null(group_controls[[as.character(grp)]])) group_controls[[as.character(grp)]] else NULL

      pc_mp_loc <- if (!is.null(gc_loc)) gc_loc$pc_mp else pc_mp
      nc_mp_loc <- if (!is.null(gc_loc)) gc_loc$nc_mp else nc_mp
      pc_ttt_loc <- if (!is.null(gc_loc)) gc_loc$pc_ttt else pc_ttt
      nc_ttt_loc <- if (!is.null(gc_loc)) gc_loc$nc_ttt else nc_ttt
      ct_loc <- if (!is.null(gc_loc)) gc_loc$crossing_threshold else crossing_threshold

      # RAF inputs
      trt_raf <- if (has_raf) df$RAF[idx] else NULL
      trt_mp_raf <- if (has_raf) df$MP[idx] else NULL
      pc_rat <- if (has_raf) {
        if (!is.null(gc_loc) && !is.null(gc_loc$pc_raf_mp_ratios)) gc_loc$pc_raf_mp_ratios else pc_raf_mp_ratios
      } else NULL

      flags <- compute_instability_flags(
        trt_mp = trt_mp_fin,
        trt_ttt = trt_ttt_vals,
        pc_mp = pc_mp_loc, nc_mp = nc_mp_loc,
        pc_ttt = pc_ttt_loc, nc_ttt = nc_ttt_loc,
        trt_raf = trt_raf, trt_mp_raf = trt_mp_raf,
        pc_raf_mp_ratios = pc_rat,
        crossing_threshold = ct_loc,
        strictness = instability_strictness,
        has_raf = has_raf
      )

      instability_summary[[trt]] <- flags

      if (flags$unstable) {
        df$matrix_instability[idx] <- TRUE
        df$trt_classification[idx] <- "INCONCLUSIVE_MATRIX_EFFECT"
        df$trt_confidence[idx] <- "LOW"

        if (!is.null(trt_summary_df) && trt %in% trt_summary_df$Treatment) {
          trt_summary_df$classification[trt_summary_df$Treatment == trt] <- "INCONCLUSIVE_MATRIX_EFFECT"
          trt_summary_df$confidence[trt_summary_df$Treatment == trt] <- "LOW"
        }

        if (verbose) {
          message(sprintf("    [UNSTABLE] %s: %d/%d flags (%s)",
                          trt, flags$n_flags, flags$min_flags_required,
                          paste(flags$reasons, collapse = "; ")))
        }
      }
    }

    n_unstable <- sum(vapply(instability_summary, function(x) x$unstable, logical(1)))
    if (verbose) {
      message(sprintf("    %d/%d treatments flagged as matrix-unstable",
                      n_unstable, length(treatments)))
    }
  }

  # ── Results Summary ───────────────────────────────────────────────────
  if (verbose) {
    samples <- df[df$Type == "Sample", ]

    message("\n\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550")
    message("KWELA 1.0.0 RESULTS")
    message("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550")
    message(sprintf("  Profile: %s | Consensus: %s | Cutoff: %.2f",
                    toupper(profile), consensus, optimal_cutoff))

    message("\n  Well-level classification:")
    print(table(samples$classification, useNA = "ifany"))

    if (!is.null(trt_summary_df) && nrow(trt_summary_df) > 0) {
      message("\n  Treatment-level consensus:")
      print(table(trt_summary_df$classification, useNA = "ifany"))
    }

    spiked <- samples[samples$is_spiked, ]
    if (nrow(spiked) > 0) {
      n_well_pos <- sum(grepl("POSITIVE", spiked$classification), na.rm = TRUE)
      message(sprintf("\n  Spiked well recovery: %d/%d (%.1f%%)",
                      n_well_pos, nrow(spiked), 100 * n_well_pos / nrow(spiked)))

      spiked_trts <- unique(spiked$Treatment)
      if (!is.null(trt_summary_df)) {
        spiked_trt_df <- trt_summary_df[trt_summary_df$is_spiked, ]
        if (nrow(spiked_trt_df) > 0) {
          n_trt_pos <- sum(spiked_trt_df$classification == "POSITIVE")
          message(sprintf("  Spiked treatment recovery (consensus): %d/%d (%.1f%%)",
                          n_trt_pos, nrow(spiked_trt_df),
                          100 * n_trt_pos / nrow(spiked_trt_df)))
        }
      }
    }

    unspiked <- samples[!samples$is_spiked, ]
    if (nrow(unspiked) > 0) {
      n_fp <- sum(grepl("POSITIVE", unspiked$classification), na.rm = TRUE)
      message(sprintf("  Unspiked FP rate (well): %d/%d (%.1f%%)",
                      n_fp, nrow(unspiked), 100 * n_fp / nrow(unspiked)))
    }

    message("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550")
  }

  # Store metadata
  attr(df, "pc_stats") <- pc_stats
  attr(df, "nc_stats") <- nc_stats
  attr(df, "separation") <- separation
  attr(df, "profile") <- profile
  attr(df, "consensus") <- consensus
  attr(df, "optimal_cutoff") <- optimal_cutoff
  attr(df, "youden_j") <- best_youden
  attr(df, "thresholds") <- list(crossing = crossing_threshold, signal = signal_threshold)
  attr(df, "trt_summary") <- trt_summary_df
  attr(df, "control_check") <- list(inverted = controls_inverted, degenerate = controls_degenerate)
  attr(df, "score_semantics") <- switch(profile,
    "standard" = "well_score = max(TTT_percentile, MPR_binary[0/1], RAF_zscore). Scale: 0-1, binary-dominated.",
    "sensitive" = "well_score = min(TTT_sigmoid, zscore_mad_MP, stochastic, RAF_concordance). Scale: 0-1, conservative.",
    "matrix_robust" = "well_score = min(MPR_continuous, RAF_ratio). Scale: 0-1, signal-dominant, TTT_weight=0.",
    "well_score scale is profile-specific and not comparable across profiles."
  )
  attr(df, "version") <- "1.0.0"
  attr(df, "mode") <- mode
  attr(df, "instability_check") <- instability_check
  attr(df, "instability_strictness") <- instability_strictness
  attr(df, "instability_summary") <- instability_summary
  attr(df, "group_controls") <- group_controls
  attr(df, "group_profiles") <- group_profiles
  attr(df, "group_separation") <- group_separation

  return(df)
}
