% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prior_post.R
\name{prior_posterior}
\alias{prior_posterior}
\alias{combine_prior_posterior}
\alias{plot_prior_posterior}
\alias{compute_prior_influence}
\alias{plot_prior_influence}
\alias{check_model_sensitivity}
\title{Compare prior to posterior}
\usage{
combine_prior_posterior(prior, post, pars = NULL, match_exact = TRUE)

plot_prior_posterior(
  prior,
  post,
  pars = NULL,
  match_exact = TRUE,
  lb = "5\%",
  ub = "95\%"
)

compute_prior_influence(
  prior,
  post,
  pars = NULL,
  match_exact = TRUE,
  remove_index_prior = TRUE
)

plot_prior_influence(prior, post, pars = NULL, match_exact = TRUE)

check_model_sensitivity(prior, post, pars = NULL)
}
\arguments{
\item{prior}{Dataframe of prior parameter estimates.
The dataframe is expected to have columns \code{Variable}, \code{Mean.}
For \code{plot_prior_posterior()}, the columns \code{lb} and \code{ub} should also be present.
For \code{compute_prior_influence()} and \code{plot_prior_influence()}, the columns \code{Index} and \code{sd} should also be present.}

\item{post}{Dataframe of posterior parameter estimates, with same columns as \code{prior}.}

\item{pars}{Vector of parameter names to plot. Defaults to all parameters presents in \code{post} and \code{prior.}}

\item{match_exact}{Logical indicating whether parameters should be matched exactly (e.g. \code{p} does not match \verb{p\[1\]}).}

\item{lb}{Name of the column in \code{prior} and \code{post} corresponding to lower bound of error bar}

\item{ub}{Name of the column in \code{prior} and \code{post} corresponding to upper bound of error bar}

\item{remove_index_prior}{Whether to remove the index variable for \code{prior} except the first one.
This is useful if a parameter with multiple index have the same prior distribution
(e.g. with subject parameters, when \code{prior} does not contain as many subjects as post for computational reasons).}
}
\value{
\itemize{
\item \code{combine_prior_posterior} returns a dataframe with the same columns as in prior and post and a column \code{Distribution}.
\item \code{compute_prior_influence} returns a dataframe with columns: \code{Variable}, \code{Index}, \code{PostShrinkage}, \code{DistPrior.}
\item \code{plot_prior_posterior} and \code{plot_prior_influence} returns a ggplot object.
}
}
\description{
\itemize{
\item \code{combine_prior_posterior} subsets and binds the prior and posterior dataframes.
\item \code{plot_prior_posterior} plots posterior CI alongside prior CI.
\item \code{compute_prior_influence} computes diagnostics of how the posterior is influenced by the prior.
\item \code{plot_prior_influence} plots diagnostics from \code{compute_prior_influence}.
\code{check_model_sensitivity} is a deprecated alias of \code{plot_prior_influence}.
}
}
\details{
\itemize{
\item Posterior shrinkage (\code{PostShrinkage = 1 - Var(Post) / Var(Prior)}), capturing how much the model is learning.
Shrinkage near 0 indicates that the data provides little information beyond the prior.
Shrinkage near 1 indicates that the data is much more informative than the prior.
\item 'Mahalanobis' distance between the mean posterior and the prior (\code{DistPrior}), capturing whether the prior "includes" the posterior.
}
}
\section{Note}{

For \code{plot_prior_posterior}, parameters with the same name but different indices are plotted together.
If their prior distribution is the same, it can be useful to only keep one index in \code{prior}.
If not, we can use \code{match_exact = FALSE} to plot \code{parameter[1]} and \code{parameter[2]} separately.
}

\examples{

library(dplyr)

prior <- data.frame(
  Variable = c("a", "b"),
  Mean = c(0, 0),
  sd = c(10, 5),
  Index = c(NA, NA)
) \%>\%
  mutate(
    `5\%` = qnorm(.05, mean = Mean, sd = sd),
    `95\%` = qnorm(.95, mean = Mean, sd = sd)
  )

post <- data.frame(
  Variable = c("a", "a", "b"),
  Mean = c(-1, 1, 2),
  sd = c(3, 4, 1),
  Index = c(1, 2, NA)
) \%>\%
  mutate(
    `5\%` = qnorm(.05, mean = Mean, sd = sd),
    `95\%` = qnorm(.95, mean = Mean, sd = sd)
  )

plot_prior_posterior(prior, post)

plot_prior_influence(prior, post, pars = c("a", "b"))

}
\references{
\href{https://betanalpha.github.io/assets/case_studies/principled_bayesian_workflow.html}{M. Betancourt, “Towards a Principled Bayesian Workflow”}, 2018.
}
