confintHSMM <- function(x, HSMM, obsdist, dwelldist, level = 0.95, B = 100, M = NA,
                        shift = FALSE, maxiter = 100, tol = 1e-5, verbose = TRUE, seed = NULL) {
  # Set seed if provided
  if (!is.null(seed)) {
    set.seed(seed)
  }

  # Extract model dimensions and parameters
  n <- length(x)
  J <- nrow(HSMM$Pi)
  Pi <- HSMM$Pi
  obspar <- HSMM$observationparameters
  dwellpar <- HSMM$dwellparameters

  # Get parameter names for labeling bootstrap results
  obs_names <- names(unlist(obspar))
  dwell_names <- names(unlist(dwellpar))

  # Initialize bootstrap storage matrices
  obs_boot <- matrix(NA, nrow = B, ncol = length(obs_names))
  colnames(obs_boot) <- obs_names
  dwell_boot <- matrix(NA, nrow = B, ncol = length(dwell_names))
  colnames(dwell_boot) <- dwell_names
  Pi_boot <- array(NA, dim = c(B, J, J))

  # Bootstrap loop with retry mechanism for failed fits
  b <- 1
  attempts <- 0
  while (b <= B) {
    attempts <- attempts + 1
    if (verbose) {
      message(sprintf("Bootstrap replicate %d of %d (attempt %d)", b, B, attempts))
    }

    # Deep copy transition matrix to avoid modification
    Pi2 <- unserialize(serialize(Pi, NULL))

    # Generate bootstrap sample using current parameter estimates
    tryCatch({sim <- generateHSMM(
      n = n, J = J, obsdist = obsdist, dwelldist = dwelldist,
      obspar = obspar, dwellpar = dwellpar, Pi = Pi2, shift = shift
    )}, error=function(e) NULL)

    # Fit HSMM to bootstrap sample with error handling
    invisible(capture.output(fit <- tryCatch({
      findmleHSMM(
        x = sim$x, J = J, M = M, obsdist = obsdist, dwelldist = dwelldist,
        obspar = obspar, dwellpar = dwellpar, Pi = Pi2,
        maxiter = maxiter, tol = tol, shift = shift, verbose=verbose
      )
    }, error = function(e) NULL)))

    # Store results if fit succeeded, otherwise retry
    if (!is.null(fit)) {
      obs_boot[b, ] <- unlist(fit$observationparameters)
      dwell_boot[b, ] <- unlist(fit$dwellparameters)
      Pi_boot[b, , ] <- fit$Pi
      b <- b + 1
    } else {
      if (verbose) message("Replicate failed - retrying...")
    }
  }

  # Check how many bootstrap replicates succeeded
  successful_boots <- sum(complete.cases(obs_boot))
  if (successful_boots < 0.8 * B) {
    warning(sprintf("Only %d of %d bootstrap replicates succeeded", successful_boots, B))
  }

  # Calculate confidence interval bounds
  alpha <- 1 - level

  # Observation parameter confidence intervals
  obs_ci <- t(apply(obs_boot, 2, quantile, probs = c(alpha / 2, 1 - alpha / 2), na.rm = TRUE))
  colnames(obs_ci) <- c("Lower", "Upper")

  # Dwell parameter confidence intervals
  dwell_ci <- t(apply(dwell_boot, 2, quantile, probs = c(alpha / 2, 1 - alpha / 2), na.rm = TRUE))
  colnames(dwell_ci) <- c("Lower", "Upper")

  # Transition matrix confidence intervals (element-wise)
  Pi_lower <- matrix(NA, J, J)
  Pi_upper <- matrix(NA, J, J)
  for (j in 1:J) {
    for (k in 1:J) {
      vals <- Pi_boot[, j, k]
      Pi_lower[j, k] <- quantile(vals, probs = alpha / 2, na.rm = TRUE)
      Pi_upper[j, k] <- quantile(vals, probs = 1 - alpha / 2, na.rm = TRUE)
    }
  }

  # Return structured results with estimates and confidence intervals
  return(list(
    obspar_CI = data.frame(
      Parameter = colnames(obs_boot),
      Estimate = as.numeric(unlist(obspar)),
      Lower = obs_ci[, 1],
      Upper = obs_ci[, 2],
      row.names = NULL
    ),
    dwellpar_CI = data.frame(
      Parameter = colnames(dwell_boot),
      Estimate = as.numeric(unlist(dwellpar)),
      Lower = dwell_ci[, 1],
      Upper = dwell_ci[, 2],
      row.names = NULL
    ),
    Pi_CI = list(
      Estimate = Pi,
      Lower = Pi_lower,
      Upper = Pi_upper
    ),
    bootstrap_samples = list(
      obspar = obs_boot,
      dwellpar = dwell_boot,
      Pi = Pi_boot,
      n_successful = successful_boots,
      n_attempts = attempts
    )
  ))
}
