#'@title Simulation of generalized Ornstein-Uhlenbeck (GOU) process
#'
#'@description Function to simulate exact N+K+1 values with change point after N+K_star, with  K_star = floor(N*t_star), for a GOU process. Starting point is 0.
#'
#'@param T1          Last time of observation
#'@param N           Number of observations on from on interval (0,T1]
#'@param t_star      Time of change-point after T1
#'@param K           Number of observation after change-point
#'@param theta       list of parameters before change-point: cos coefficients (>=1), sine and sigma
#'@param theta_star  list of parameters after  change-point: cos coefficients (>=1), sine and sigma
#'@param sigma       volatility parameter of the GOU process
#'@return \item{X}{Simulated path evaluated at points k x T1/N, 0 <= k <= N+K}
#'
#'@examples
#' set.seed(3253)
#' T1=20
#' N=500
#' K=2*N
#' t_star=0
#' theta=list(cos=c(1,2),alpha=1) # d=3 parameters for the drift
#' theta_star=list(cos=c(2,5),alpha=1)
#' sigma=3
#' X=SimGOUexact(T1,N,t_star,K,theta,theta_star,sigma)
#'@export



SimGOUexact <- function(T1,N,t_star=0,K,theta,theta_star,sigma)
{

 param = c(theta$cos,theta$sin)
 alpha = theta$alpha
 param_star = c(theta_star$cos,theta_star$sin)
 alpha_star = theta_star$alpha
 p1 = length(theta$cos)
 p = length(param) # does not contains alpha

 q = p-p1 # number of non zero sine coefficients

 DeltaN = T1/N # time increment Delta t

 K_star = floor(N*t_star)
  tt = c(0:(N+K))*DeltaN

  eps= sigma*rnorm(N+K)
  ind2 = N+K_star+ c(1:(K-K_star))
  eps1 = eps[1:(N+K_star)]*sqrt( (1-exp(-2*alpha*DeltaN))/(2*alpha))
  eps2 = eps[ind2]*sqrt( (1-exp(-2*alpha_star*DeltaN))/(2*alpha_star))

  eps=c(eps1,eps2)
  dd=c(1:p)-1

  c = alpha^2+4*pi^2*dd^2
  a = alpha/c
  b = 2*pi*dd/c
  H = exp(-alpha*DeltaN)
  A = a*cos(2*pi*dd*DeltaN)+b*sin(2*pi*dd*DeltaN)-a*H
  B = b*H - b*cos(2*pi*dd*DeltaN)+a*sin(2*pi*dd*DeltaN)

  H_star = exp(-alpha_star*DeltaN)
  a_star = alpha_star/(alpha_star^2+4*pi^2*dd^2)
  b_star = 2*pi*dd/(alpha_star^2+4*pi^2*dd^2)
  A_star = a_star*cos(2*pi*dd*DeltaN)+b_star*sin(2*pi*dd*DeltaN)-a_star*H_star
  B_star = b_star*H_star - b_star*cos(2*pi*dd*DeltaN)+a_star*sin(2*pi*dd*DeltaN)



  phi=matrix(1,ncol=1,nrow=(N+K))
  if(p>1)
  {
    for(k in 2:p)
     {
      phi=cbind(phi,fcos(tt[-(N+K)]*(k-1) ))
     }
  }
  psi=NULL
  for(k in 1:p)
    {
      psi=cbind(psi,fsin(tt[-(N+K)]*(k-1) ))
    }


  Phi=matrix(0,ncol=p1,nrow=(N+K))


  for(k in 1:p1)
  {
    Phi[,k] = A[k]*phi[,k]-B[k]*psi[,k]
  }

  Psi=NULL
  if(q>0)
  {
    Psi=matrix(0,ncol=q,nrow=(N+K))
     for(k in 1:q)
      {
        Psi[,k] = B[k+1]*phi[,(k+1)]+A[k+1]*psi[,(k+1)]
      }
  }


  Phi_star=matrix(0,ncol=p1,nrow=(N+K))


  for(k in 1:p1)
  {
    Phi_star[,k] = A_star[k]*phi[,k]-B_star[k]*psi[,k]
  }

  Psi_star=NULL
  if(q>0)
  {
    Psi_star=matrix(0,ncol=q,nrow=(N+K))
    for(k in 1:q)
    {
      Psi_star[,k] = B_star[k+1]*phi[,(k+1)]+A_star[k+1]*psi[,(k+1)]
    }
  }

  X=rep(0,(N+K))

  x=0
  for( i in 1:(N+K_star))
  {
    X[i] = H*x + sum(param*c(Phi[i,],Psi[i,]) )  + eps[i]
    x = X[i]
  }

  for(i in (N+K_star+1):(N+K))
  {
    X[i] = H_star*x + sum(param_star*c(Phi_star[i,],Psi_star[i,]))   + eps[i]
    x = X[i]
  }

  X=c(0,X)
}
