#' Function to perform an EM algorithm (multivariate normal with shared covariance (parametric, correlation-aware))
#' for a 2-component mixture where each observation \eqn{i} has a p-value \eqn{p_i}, and an auxiliary feature vector \eqn{v_i}.
#' @param pvalues Vector of gene-level p-values.
#' @param v An auxiliary feature vector.
#' @param alpha0 Mixing proportion (prior probability) that an observation belongs to the alternative/signal component \eqn{Z = 1}
#'   in the two-component mixture; typically \eqn{\pi_1 = 1 - \pi_0}.
#' @param alpha1 Shape parameter of the alternative/signal p-value distribution, modeled as
#'   \eqn{p \sim \mathrm{Beta}(\alpha_1, 1)} when \eqn{Z = 1}; values \eqn{< 1} concentrate mass near 0 (enrichment of small p-values).
#' @param pi0 Mixing proportion (prior probability) that an observation belongs to the null/background component \eqn{Z = 0}
#'   in the two-component mixture.
#' @param pi1 Mixing proportion (prior probability) that an observation belongs to the alternative/signal component \eqn{Z = 1}
#'   in the two-component mixture; typically \eqn{\pi_1 = 1 - \pi_0}.
#' @param max.it Maximum number of iterations for the EM algorithm.
#' @param verbose A logical value defining if the iteration information should be printed or not.
#' @return Returns the posterior probability of association for each gene.
#' @name model.MVN
#' @keywords internal
model.MVN <- function(pvalues,v, alpha0, alpha1,pi0,pi1, max.it, verbose=verbose){

  n <- dim(v)[1]
  d <- dim(v)[2]
  
  r <- EMinfer(pvalues,pi0,pi1, alpha0, alpha1, max.it)
  alpha0 <- r$alpha0
  alpha1 <- r$alpha1
  pi <- r$pi
  mu0 <- seq(0,0,length.out = d)
  mu1 <- seq(0,0,length.out = d)
  sigma <- diag(d)
  z <- seq(0,0,length.out = n)
  
  loglik.history <- c()
  i <- 0
  while(1){
    i <- i + 1
    
    tmp0 <- log(1-pi) + log(alpha0) + (alpha0-1)*log(pvalues) - 1/2 * (sum(log(det(sigma))) + apply( (v-matrix(rep(mu0,n),nrow=n,byrow = T)) %*% solve(sigma) * (v-matrix(rep(mu0,n),nrow=n,byrow = T)),1,sum) )
    tmp1 <- log(pi) + log(alpha1) + (alpha1-1)*log(pvalues) - 1/2 * (sum(log(det(sigma) )) + apply( (v-matrix(rep(mu1,n),nrow=n,byrow = T)) %*% solve(sigma) * (v-matrix(rep(mu1,n),nrow=n,byrow = T)),1,sum) )
    z <- 1 / (1 + exp(tmp0 - tmp1))
    
    
    max.tmp <- max(c(tmp1,tmp0))
    loglik <- sum(log(exp(tmp1-max.tmp) + exp(tmp0-max.tmp) + 1e-320) + max.tmp)
    loglik.history <- c(loglik.history,loglik)
    
    pi <- sum(z) / n
    alpha0 <- - sum(1-z) / sum((1-z) * log(pvalues))
    alpha1 <- - sum(z) / sum(z * log(pvalues))
    
    mu0 <- apply(v * (1-z), 2, sum) / sum(1-z)
    mu1 <- apply(v * z, 2, sum) / sum(z)
    
    sigma <- 1/n*( t( z * (v - matrix(rep(mu1,n),nrow=n,byrow = T))) %*% (v - matrix(rep(mu1,n),nrow=n,byrow = T))  + t( (1-z) * (v - matrix(rep(mu0,n),nrow=n,byrow = T))) %*% (v - matrix(rep(mu0,n),nrow=n,byrow = T)) )
    
    if(verbose) print(sprintf('iter %02d , log(likelihood) %.6f',i,loglik))
    
    if(i > 10 && abs(loglik.history[i-10] - loglik.history[i]) < 1e-6 ) break;
    if(i >= 5000) break;
    if(i > 1 && loglik.history[i] < loglik.history[i-1]) break;
  }
  
  result <- list(loglik=loglik.history, post=z, alpha0=alpha0,alpha1=alpha1,pi=pi,mu0=mu0,mu1=mu1,sigma=sigma)
  return(result)
}