% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_clusters.R
\name{plot_clusters}
\alias{plot_clusters}
\title{Plot Clustered Individual Networks}
\usage{
plot_clusters(
  dynEGA.object,
  clustering,
  include,
  type = c("population", "average"),
  node.size = 3,
  label.size = 1.2,
  ...
)
}
\arguments{
\item{dynEGA.object}{A \code{\link[EGAnet]{dynEGA}} or a
\code{\link[EGAnet]{dynEGA.ind.pop}} object}

\item{clustering}{Vector (length of individuals).
A vector of cluster membership for each individual in the \code{dynEGA.object}.
A common and easy option is to use an \code{infoCluster} object containing cluster
assignments for individuals (see \code{\link[EGAnet]{infoCluster}}).
Accepts any vector of memberships}

\item{include}{Numeric vector. Specifies which cluster memberships should be
explicitly included in the plot. By default, all clusters are shown. Use this
argument to restrict the visualization to a subset of clusters (e.g.,
`include = c(1, 3, 5)` will only display clusters 1, 3, and 5)}

\item{type}{Character (length = 1).
Type of visualization to produce:

\itemize{

\item \code{"population"} (default) --- Cluster-level population networks by stacking
derivatives across individuals in the cluster and re-estimating with \code{\link[EGAnet]{EGA}}

\item \code{"average"} --- Averaged networks with a consensus clustering matrix (pairwise)
membership similarity that is supplied to \code{\link[EGAnet]{community.detection}}

}

In either type, the argument setting applied in your `dynEGA.object` will be carried
forward into the network estimation (`"population"`) and community detection (both)}

\item{node.size}{Numeric (length = 1 or number of variables).
Node size in network visualizations.
Defaults to \code{3}}

\item{label.size}{Numeric (length = 1 or number of nodes.
Label size in network visualizations.
Defaults to \code{1.2}}

\item{...}{Additional arguments passed to
\code{\link[EGAnet]{EGA}} and
\code{\link[EGAnet]{compare.EGA.plots}}}
}
\value{
Plots created using \code{\link[EGAnet]{compare.EGA.plots}}:

\item{"population"}{A plot combining cluster-level population network plots}

\item{"average"}{A plot combining cluster-average network plots}
}
\description{
Visualize clusters of individual networks identified by
\code{\link[EGAnet]{infoCluster}} (or another clustering method) applied to a
\code{\link[EGAnet]{dynEGA}} object

Provides visualization modes:
\itemize{
  \item \code{"population"}: Cluster-level "population" networks, obtained by stacking derivatives across individuals and estimating a new EGA
  \item \code{"average"}: Cluster-average networks, obtained by averaging adjacency matrices across individuals in the cluster
}
}
\details{
This function provides flexible visualization of individual clusters obtained from
\code{\link[EGAnet]{infoCluster}} (or other clustering input; see examples):

\itemize{

\item Cluster-level population networks treat each cluster as a "mini-population," stacking
derivatives across individuals to re-estimate an EGA

\item Average networks show the mean connectivity structure for each cluster,
with consensus clustering memberships

}

Function automatically extracts the appropriate derivatives (\code{"population"}) and
networks (\code{"average"}) from the \code{dynEGA.object} results.
}
\examples{
# Load data
data <- sim.dynEGA

\dontrun{
# Run dynEGA with optimization
optimized_all <- dynEGA(
  data = data, level = c("individual", "population"),
  n.embed = 3:25, n.embed.optimize = TRUE
)

# Cluster individuals
clust <- infoCluster(dynEGA.object = optimized_all)

# Cluster-level population networks
plot_clusters(
  dynEGA.object = optimized_all,
  clustering = clust,
  type = "population"
)

# Average networks per cluster
plot_clusters(
  dynEGA.object = optimized_all,
  clustering = clust,
  type = "average"
)

# Cluster-level population networks, including only Cluster 2:
plot_clusters(
  dynEGA.object = optimized_all,
  clustering = clust, include = 2,
  type = "population"
)

# Using alternative clusters
plot_clusters(
  dynEGA.object = optimized_all,
  clustering = rep(1:2, each = 50), # vector of memberships
  type = "population"
)

# Run with non-optimized dynEGA
standard_all <- dynEGA(
  data = data,
  level = c("individual", "population")
)

# Obtain clusters
clust_standard <- infoCluster(dynEGA.object = standard_all)

# Plot clusters with population
plot_clusters(
  dynEGA.object = standard_all,
  clustering = clust_standard,
  type = "population"
)}

}
\seealso{
\code{\link[EGAnet]{dynEGA}}, \code{\link[EGAnet]{infoCluster}}, \code{\link[EGAnet]{EGA}},
\code{\link[EGAnet]{compare.EGA.plots}}
}
\author{
Hudson Golino <hfg9s at virginia.edu>
}
