% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_laus.R
\name{get_laus}
\alias{get_laus}
\title{Download Local Area Unemployment Statistics (LAUS) Data}
\usage{
get_laus(
  geography = "state_adjusted",
  monthly_only = TRUE,
  transform = TRUE,
  suppress_warnings = TRUE,
  return_diagnostics = FALSE
)
}
\arguments{
\item{geography}{Character string specifying the geographic level and adjustment type.
Default is "state_adjusted". Valid options are:
\itemize{
  \item "state_current_adjusted" - Current seasonally adjusted state data
  \item "state_unadjusted" - All historical unadjusted state data
  \item "state_adjusted" - All historical seasonally adjusted state data (default)
  \item "region_unadjusted" - Unadjusted regional and division data
  \item "region_adjusted" - Seasonally adjusted regional and division data
  \item "metro" - Metropolitan statistical area data
  \item "division" - Division-level data
  \item "micro" - Micropolitan statistical area data
  \item "combined" - Combined statistical area data
  \item "county" - County-level data (large file >300MB)
  \item "city" - City and town data (large file >300MB)
  \item "1990-1994" - Comprehensive unadjusted data for 1990-1994
  \item "1995-1999" - Comprehensive unadjusted data for 1995-1999
  \item "2000-2004" - Comprehensive unadjusted data for 2000-2004
  \item "2005-2009" - Comprehensive unadjusted data for 2005-2009
  \item "2010-2014" - Comprehensive unadjusted data for 2010-2014
  \item "2015-2019" - Comprehensive unadjusted data for 2015-2019
  \item "2020-2024" - Comprehensive unadjusted data for 2020-2024
  \item "2025-2029" - Comprehensive unadjusted data for 2025-2029
  \item "ST" - Any state two-character USPS abbreviation, plus DC and PR
}}

\item{monthly_only}{Logical. If TRUE (default), excludes annual data (period M13)
and creates a date column from year and period.}

\item{transform}{Logical. If TRUE (default), converts rate and ratio measures from
percentages to proportions by dividing by 100. Unemployment rates will be expressed
as decimals (e.g., 0.05 for 5\% unemployment) rather than as whole numbers (e.g. 5).}

\item{suppress_warnings}{Logical. If TRUE (default), suppress individual download warnings and diagnostic messages
for cleaner output during batch processing. If FALSE, returns the data and prints warnings and messages to the console.}

\item{return_diagnostics}{Logical. If TRUE, returns a bls_data_collection object
with full diagnostics. If FALSE (default), returns just the data table.}
}
\value{
By default, returns a data.table with LAUS data. If return_diagnostics = TRUE,
  returns a bls_data_collection object containing LAUS data with the following key columns:
  \describe{
    \item{series_id}{BLS series identifier}
    \item{year}{Year of observation}
    \item{period}{Time period (M01-M12 for months, M13 for annual)}
    \item{value}{Employment statistic value (transformed if transform = TRUE)}
    \item{date}{Date of observation (if monthly_only = TRUE)}
    \item{area_text}{Geographic area name}
    \item{area_type_code}{Code indicating area type}
    \item{measure_text}{Type of measure (unemployment rate, labor force, employment, etc.)}
    \item{seasonal}{Seasonal adjustment status}
  }
}
\description{
This function downloads Local Area Unemployment Statistics data from the U.S. Bureau
 of Labor Statistics. Due to the large size of some LAUS datasets (county and city
 files are >300MB), users must specify which geographic level to download. The function
 provides access to both seasonally adjusted and unadjusted data at various geographic levels. 
 Additional datasets provide comprehensive non-seasonally-adjusted data for all areas broken out 
 in 5-year increments
}
\details{
The function joins data from multiple BLS files:
  \itemize{
    \item Main data file (varies by geography selection)
    \item Series definitions (la.series)
    \item Area codes and names (la.area)
    \item Measure definitions (la.measure)
  }
}
\examples{
\donttest{
# Download state-level seasonally adjusted data (default operation)
laus_states <- get_laus()

# View unemployment rates by state for latest period
unemployment <- laus_states[grepl("unemployment rate", measure_text) & date == max(date)]

# Download unadjusted state data
laus_states_unadjusted <- get_laus("state_unadjusted")

# Download metro area data with rates as whole number percentages (64.3 instead of 0.643)
laus_metro <- get_laus("metro", transform = FALSE)

# Get full diagnostic object if needed
laus_with_diagnostics <- get_laus(return_diagnostics = TRUE)
print_bls_warnings(laus_with_diagnostics)


}
}
