## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  warning = FALSE,
  message = FALSE
)

## ----setup--------------------------------------------------------------------
library(phinterval)
library(lubridate, warn.conflicts = FALSE)
library(dplyr, warn.conflicts = FALSE)
library(tidyr, warn.conflicts = FALSE)

## -----------------------------------------------------------------------------
jan <- interval(ymd("2025-01-01"), ymd("2025-02-01"))
nov <- interval(ymd("2025-11-01"), ymd("2025-12-01"))
full_2025 <- interval(ymd("2025-01-01"), ymd("2026-01-01"))

## -----------------------------------------------------------------------------
lubridate::intersect(jan, nov)

phint_intersect(jan, nov)

## -----------------------------------------------------------------------------
lubridate::intersect(jan, nov) / duration(days = 1)

phint_intersect(jan, nov) / duration(days = 1)

## -----------------------------------------------------------------------------
try(lubridate::setdiff(full_2025, nov))

phint_setdiff(full_2025, nov)

## -----------------------------------------------------------------------------
lubridate::union(jan, nov)

phint_union(jan, nov)

## -----------------------------------------------------------------------------
lubridate::setdiff(jan, jan)

phint_setdiff(jan, jan)

## -----------------------------------------------------------------------------
jobs <- tribble(
  ~name,   ~job_title,             ~start,        ~end,
  "Greg",  "Mascot",               "2018-01-01",  "2018-06-03",
  "Greg",  "Executive Assistant",  "2018-06-10",  "2020-04-01",
  "Greg",  "Chief of Staff",       "2020-03-01",  "2020-11-28",
  "Tom",   "Chairman",             "2019-05-01",  "2020-11-10",
  "Tom",   "CEO",                  "2020-11-10",  "2020-12-31",
  "Shiv",  "Political Consultant", "2017-01-01",  "2019-04-01"
)

## -----------------------------------------------------------------------------
vacation <- interval(ymd("2017-12-23"), ymd("2017-12-29"))

## -----------------------------------------------------------------------------
try(
  jobs |>
    mutate(
      span = interval(start, end),
      span = setdiff(span, vacation)
    ) |>
    select(name, job_title, span)
)

## -----------------------------------------------------------------------------
jobs |>
  mutate(
    span = interval(start, end),
    span = phint_setdiff(span, vacation)
  ) |>
  select(name, job_title, span)

## ----include = FALSE--------------------------------------------------------------------------------------------------
opts <- options(width = 120)

## ---------------------------------------------------------------------------------------------------------------------
employment <- jobs |>
  mutate(span = interval(start, end)) |>
  group_by(name) |>
  summarize(employed = phint_squash(span))

employment

## ---------------------------------------------------------------------------------------------------------------------
datetime_squash(
  start = ymd(jobs$start),
  end = ymd(jobs$end),
  by = jobs$name,
  keep_by = TRUE,
  order_by = TRUE
)

## ----include = FALSE----------------------------------------------------------
options(opts)

## -----------------------------------------------------------------------------
employment |>
  reframe(phint_unnest(employed, key = name))

## -----------------------------------------------------------------------------
unemployment <- employment |>
  mutate(
    # Find the gaps between jobs
    unemployed = phint_invert(employed),
    
    # Calculate duration of unemployment
    days_unemployed = unemployed / ddays(1)
  ) |>
  select(name, unemployed, days_unemployed)

unemployment

## -----------------------------------------------------------------------------
monday <- interval(ymd("2025-11-10"), ymd("2025-11-11"))
tuesday <- interval(ymd("2025-11-11"), ymd("2025-11-12"))

## -----------------------------------------------------------------------------
midnight_monday <- ymd_hms("2025-11-11 00:00:00")
phint_within(midnight_monday, monday)
phint_within(midnight_monday, tuesday)

## -----------------------------------------------------------------------------
phint_intersect(monday, tuesday) == as_phinterval(midnight_monday)

## -----------------------------------------------------------------------------
not_monday <- phint_complement(monday)
not_monday

phint_intersect(monday, not_monday)

## -----------------------------------------------------------------------------
phint_overlaps(monday, tuesday, bounds = "()")
phint_intersect(monday, tuesday, bounds = "()")

## -----------------------------------------------------------------------------
monday_at_9AM <- as_phinterval(ymd_hms("2025-11-10 00:09:00"))
phint_within(monday_at_9AM, monday, bounds = "()")
phint_within(midnight_monday, monday, bounds = "()")

## -----------------------------------------------------------------------------
phint <- phint_squash(c(monday_at_9AM, tuesday))
phint

phint_sift(phint)

## -----------------------------------------------------------------------------
monday_noon <- as_phinterval(ymd_hms("2025-11-10 12:00:00"))
monday_lunch_break <- interval(
  ymd_hms("2025-11-10 12:00:00"), 
  ymd_hms("2025-11-10 13:00:00")
)

phint_setdiff(monday, monday_lunch_break) # Removes a non-zero interval
phint_setdiff(monday, monday_noon)        # Instantaneous - no effect

## -----------------------------------------------------------------------------
intvl <- interval(ymd("2020-01-01"), ymd("2020-01-02"), tzone = "nozone")
phint <- phinterval(ymd("2020-01-01"), ymd("2020-01-02"), tzone = "nozone")
intvl == phint

## ----include = FALSE----------------------------------------------------------
rlang::reset_warning_verbosity("phinterval_warning_unrecognized_tzone")

## -----------------------------------------------------------------------------
print(phint)

## -----------------------------------------------------------------------------
is_recognized_tzone("America/New_York")
is_recognized_tzone("nozone")
is_recognized_tzone(NA_character_)

## -----------------------------------------------------------------------------
na_zoned <- as.POSIXct("2021-01-01", tz = NA_character_)
as_phinterval(na_zoned)

## -----------------------------------------------------------------------------
int_est <- interval(ymd("2020-01-01"), ymd("2020-01-02"), tzone = "EST")
int_utc <- interval(ymd("2020-01-01"), ymd("2020-01-02"), tzone = "UTC")
int_lcl <- interval(ymd("2020-01-01"), ymd("2020-01-02"), tzone = "")

phint_union(int_est, int_utc)
phint_union(int_utc, int_est)
phint_union(int_lcl, int_est)

## -----------------------------------------------------------------------------
span <- phinterval(ymd("2000-08-05"), ymd("2000-11-29"))
date <- ymd("2021-01-01")

span == date

## -----------------------------------------------------------------------------
span == as_phinterval(date)

