/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef H5Oprivate_H
#define H5Oprivate_H

typedef struct H5O_t      H5O_t;
typedef struct H5O_fill_t H5O_fill_t;

#include "H5Opublic.h" 

#include "H5Dpublic.h" 
#include "H5Lpublic.h" 
#include "H5Spublic.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5Fprivate.h"  
#include "H5HGprivate.h" 
#include "H5SLprivate.h" 
#include "H5Tprivate.h"  
#include "H5VLprivate.h" 
#include "H5Zprivate.h"  
#include "H5RTprivate.h" 

typedef struct H5O_msg_class_t H5O_msg_class_t;
typedef struct H5O_mesg_t      H5O_mesg_t;

#define H5O_FHEAP_MAN_WIDTH            4
#define H5O_FHEAP_MAN_START_BLOCK_SIZE 1024
#define H5O_FHEAP_MAN_MAX_DIRECT_SIZE  (64 * 1024)
#define H5O_FHEAP_MAN_MAX_INDEX        40
#define H5O_FHEAP_MAN_START_ROOT_ROWS  1
#define H5O_FHEAP_CHECKSUM_DBLOCKS     true
#define H5O_FHEAP_MAX_MAN_SIZE         (4 * 1024)
#define H5O_FHEAP_ID_LEN               8

#define H5O_MESG_MAX_SIZE 65536 
#define H5O_ALL           (-1)  
#define H5O_FIRST         (-2)  

#define H5O_MSG_NO_FLAGS_SET                            0x00u
#define H5O_MSG_FLAG_CONSTANT                           0x01u
#define H5O_MSG_FLAG_SHARED                             0x02u
#define H5O_MSG_FLAG_DONTSHARE                          0x04u
#define H5O_MSG_FLAG_FAIL_IF_UNKNOWN_AND_OPEN_FOR_WRITE 0x08u
#define H5O_MSG_FLAG_MARK_IF_UNKNOWN                    0x10u
#define H5O_MSG_FLAG_WAS_UNKNOWN                        0x20u
#define H5O_MSG_FLAG_SHAREABLE                          0x40u
#define H5O_MSG_FLAG_FAIL_IF_UNKNOWN_ALWAYS             0x80u
#define H5O_MSG_FLAG_BITS                                                                                    \
    (H5O_MSG_FLAG_CONSTANT | H5O_MSG_FLAG_SHARED | H5O_MSG_FLAG_DONTSHARE |                                  \
     H5O_MSG_FLAG_FAIL_IF_UNKNOWN_AND_OPEN_FOR_WRITE | H5O_MSG_FLAG_MARK_IF_UNKNOWN |                        \
     H5O_MSG_FLAG_WAS_UNKNOWN | H5O_MSG_FLAG_SHAREABLE | H5O_MSG_FLAG_FAIL_IF_UNKNOWN_ALWAYS)

#define H5O_UPDATE_TIME  0x01u
#define H5O_UPDATE_FORCE 0x02u 

#define H5O_HASH_SIZE 32

#define H5O_CRT_ATTR_MAX_COMPACT_NAME "max compact attr"    
#define H5O_CRT_ATTR_MIN_DENSE_NAME   "min dense attr"      
#define H5O_CRT_OHDR_FLAGS_NAME       "object header flags" 
#define H5O_CRT_PIPELINE_NAME         "pline"               
#define H5O_CRT_PIPELINE_DEF                                                                                 \
    {                                                                                                        \
        {0, NULL, H5O_NULL_ID, {{0, HADDR_UNDEF}}}, H5O_PLINE_VERSION_1, 0, 0, NULL                          \
    }
#ifdef H5O_ENABLE_BOGUS
#define H5O_BOGUS_MSG_FLAGS_NAME "bogus msg flags" 
#define H5O_BOGUS_MSG_FLAGS_SIZE sizeof(uint8_t)

#define H5O_BOGUS_MSG_ID_NAME "bogus msg id" 
#define H5O_BOGUS_MSG_ID_SIZE sizeof(unsigned)

#endif 

#ifdef H5O_ENABLE_BAD_MESG_COUNT
#define H5O_BAD_MESG_COUNT_NAME "bad message count" 
#define H5O_BAD_MESG_COUNT_SIZE sizeof(bool)
#endif 

#define H5O_CPY_OPTION_NAME "copy object" 
#define H5O_CPY_MERGE_COMM_DT_LIST_NAME                                                                      \
    "merge committed dtype list" 
#define H5O_CPY_MCDT_SEARCH_CB_NAME                                                                          \
    "committed dtype list search" 

#ifdef H5O_MODULE
#define H5O_OH_GET_ADDR(O)    ((O)->chunk[0].addr)
#define H5O_OH_GET_VERSION(O) ((O)->version)
#define H5O_OH_GET_FLAGS(O)   ((O)->flags)
#define H5O_OH_GET_MTIME(O)   ((O)->mtime)
#else 
#define H5O_OH_GET_ADDR(O)    (H5O_get_oh_addr(O))
#define H5O_OH_GET_VERSION(O) (H5O_get_oh_version(O))
#define H5O_OH_GET_FLAGS(O)   (H5O_get_oh_flags(O))
#define H5O_OH_GET_MTIME(O)   (H5O_get_oh_mtime(O))
#endif 

#define H5O_UPDATE_SHARED(SH_MESG, SH_TYPE, F, MSG_TYPE, CRT_IDX, OH_ADDR)                                   \
    {                                                                                                        \
        (SH_MESG)->type          = (SH_TYPE);                                                                \
        (SH_MESG)->file          = (F);                                                                      \
        (SH_MESG)->msg_type_id   = (MSG_TYPE);                                                               \
        (SH_MESG)->u.loc.index   = (CRT_IDX);                                                                \
        (SH_MESG)->u.loc.oh_addr = (OH_ADDR);                                                                \
    } 

typedef union {
    uint8_t  id[H5O_FHEAP_ID_LEN]; 
    uint64_t val;                  
} H5O_fheap_id_t;

typedef struct H5O_loc_t {
    H5F_t  *file;         
    haddr_t addr;         
    bool    holding_file; 
} H5O_loc_t;

typedef struct H5O_copy_dtype_merge_list_t {
    char                               *path; 
    struct H5O_copy_dtype_merge_list_t *next; 
} H5O_copy_dtype_merge_list_t;

typedef struct H5O_mcdt_cb_info_t {
    H5O_mcdt_search_cb_t func;
    void                *user_data;
} H5O_mcdt_cb_info_t;

typedef struct H5O_copy_t {
    bool                         copy_shallow;           
    bool                         expand_soft_link;       
    bool                         expand_ext_link;        
    bool                         expand_ref;             
    bool                         copy_without_attr;      
    bool                         preserve_null;          
    bool                         merge_comm_dt;          
    H5O_copy_dtype_merge_list_t *dst_dt_suggestion_list; 
    int                          curr_depth;             
    int                          max_depth;              
    H5SL_t                      *map_list;               
    H5SL_t                      *dst_dt_list; 
    bool dst_dt_list_complete;      
    H5O_t *oh_dst;                  
    H5F_t *file_dst;                
                                    
    void                *shared_fo; 
    H5O_mcdt_search_cb_t mcdt_cb;   
    void *mcdt_ud;                  
    hid_t lcpl_id;                  
} H5O_copy_t;

#define H5O_NULL_ID        0x0000 
#define H5O_SDSPACE_ID     0x0001 
#define H5O_LINFO_ID       0x0002 
#define H5O_DTYPE_ID       0x0003 
#define H5O_FILL_ID        0x0004 
#define H5O_FILL_NEW_ID    0x0005 
#define H5O_LINK_ID        0x0006 
#define H5O_EFL_ID         0x0007 
#define H5O_LAYOUT_ID      0x0008 
#define H5O_BOGUS_VALID_ID 0x0009 
#define H5O_GINFO_ID       0x000a 
#define H5O_PLINE_ID       0x000b 
#define H5O_ATTR_ID        0x000c 
#define H5O_NAME_ID        0x000d 
#define H5O_MTIME_ID       0x000e 
#define H5O_SHMESG_ID      0x000f 
#define H5O_CONT_ID        0x0010 
#define H5O_STAB_ID        0x0011 
#define H5O_MTIME_NEW_ID   0x0012 
#define H5O_BTREEK_ID      0x0013 
#define H5O_DRVINFO_ID     0x0014 
#define H5O_AINFO_ID       0x0015 
#define H5O_REFCOUNT_ID    0x0016 
#define H5O_FSINFO_ID      0x0017 
#define H5O_MDCI_MSG_ID    0x0018 
#define H5O_UNKNOWN_ID     0x0019 

#define H5O_DELETED_ID 0x001a 

#define H5O_BOGUS_INVALID_ID 0x001b 

#define H5O_SHARE_TYPE_UNSHARED  0 
#define H5O_SHARE_TYPE_SOHM      1 
#define H5O_SHARE_TYPE_COMMITTED 2 
#define H5O_SHARE_TYPE_HERE      3 

#define H5O_IS_STORED_SHARED(T)                                                                              \
    ((((T) == H5O_SHARE_TYPE_SOHM) || ((T) == H5O_SHARE_TYPE_COMMITTED)) ? true : false)

#define H5O_IS_TRACKED_SHARED(T) ((T) > 0)

typedef struct H5O_mesg_loc_t {
    H5O_msg_crt_idx_t index;   
    haddr_t           oh_addr; 
} H5O_mesg_loc_t;

typedef struct H5O_shared_t {
    unsigned type;        
    H5F_t   *file;        
    unsigned msg_type_id; 
    union {
        H5O_mesg_loc_t loc;     
        H5O_fheap_id_t heap_id; 
    } u;
} H5O_shared_t;

typedef struct H5O_linfo_t {
    
    bool    track_corder;    
    bool    index_corder;    
    int64_t max_corder;      
    haddr_t corder_bt2_addr; 

    
    hsize_t nlinks;        
    haddr_t fheap_addr;    
    haddr_t name_bt2_addr; 
} H5O_linfo_t;

#define H5O_FILL_VERSION_1 1

#define H5O_FILL_VERSION_2 2

#define H5O_FILL_VERSION_3 3

#define H5O_FILL_VERSION_LATEST H5O_FILL_VERSION_3

struct H5O_fill_t {
    H5O_shared_t sh_loc; 

    unsigned         version;      
    H5T_t           *type;         
    ssize_t          size;         
    void            *buf;          
    H5D_alloc_time_t alloc_time;   
    H5D_fill_time_t  fill_time;    
    bool             fill_defined; 
};

typedef struct H5O_link_hard_t {
    haddr_t addr; 
} H5O_link_hard_t;

typedef struct H5O_link_soft_t {
    char *name; 
} H5O_link_soft_t;

typedef struct H5O_link_ud_t {
    void  *udata; 
    size_t size;  
} H5O_link_ud_t;

typedef struct H5O_link_t {
    H5L_type_t type;         
    bool       corder_valid; 
    int64_t    corder;       
    H5T_cset_t cset;         
    char      *name;         
    union {
        H5O_link_hard_t hard; 
        H5O_link_soft_t soft; 
        H5O_link_ud_t   ud;   
    } u;
} H5O_link_t;

#define H5O_EFL_ALLOC     16            
#define H5O_EFL_UNLIMITED H5F_UNLIMITED 

typedef struct H5O_efl_entry_t {
    size_t  name_offset; 
    char   *name;        
    HDoff_t offset;      
    hsize_t size;        
} H5O_efl_entry_t;

typedef struct H5O_efl_t {
    haddr_t          heap_addr; 
    size_t           nalloc;    
    size_t           nused;     
    H5O_efl_entry_t *slot;      
} H5O_efl_t;

#define H5O_LAYOUT_NDIMS (H5S_MAX_RANK + 1)

#define H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS 0x01
#define H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER         0x02
#define H5O_LAYOUT_ALL_CHUNK_FLAGS                                                                           \
    (H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS | H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER)

#define H5O_LAYOUT_VDS_GH_ENC_VERS_0 0

#define H5O_LAYOUT_VDS_GH_ENC_VERS_1 1

#define H5O_LAYOUT_VDS_SOURCE_FILE_SHARED 0x01
#define H5O_LAYOUT_VDS_SOURCE_DSET_SHARED 0x02
#define H5O_LAYOUT_VDS_SOURCE_SAME_FILE   0x04
#define H5O_LAYOUT_ALL_VDS_FLAGS                                                                             \
    (H5O_LAYOUT_VDS_SOURCE_FILE_SHARED | H5O_LAYOUT_VDS_SOURCE_DSET_SHARED | H5O_LAYOUT_VDS_SOURCE_SAME_FILE)

#define H5O_LAYOUT_VERSION_1 1

#define H5O_LAYOUT_VERSION_2 2

#define H5O_LAYOUT_VERSION_3 3

#define H5O_LAYOUT_VERSION_4 4

#define H5O_LAYOUT_VERSION_5 5

#define H5O_LAYOUT_VERSION_DEFAULT H5O_LAYOUT_VERSION_3

#define H5O_LAYOUT_VERSION_LATEST H5O_LAYOUT_VERSION_5

struct H5D_layout_ops_t; 
struct H5D_chunk_ops_t;  

typedef struct H5O_storage_contig_t {
    haddr_t addr; 
    hsize_t size; 
} H5O_storage_contig_t;

typedef struct H5O_storage_chunk_btree_t {
    haddr_t dset_ohdr_addr; 
    H5UC_t *shared;         
} H5O_storage_chunk_btree_t;

struct H5FA_t; 

typedef struct H5O_storage_chunk_farray_t {
    haddr_t        dset_ohdr_addr; 
    struct H5FA_t *fa;             
} H5O_storage_chunk_farray_t;

struct H5EA_t; 

typedef struct H5O_storage_chunk_earray_t {
    haddr_t        dset_ohdr_addr; 
    struct H5EA_t *ea;             
} H5O_storage_chunk_earray_t;

typedef struct H5O_storage_chunk_single_filt_t {
    hsize_t  nbytes;      
    uint32_t filter_mask; 
} H5O_storage_chunk_single_filt_t;

struct H5B2_t; 

typedef struct H5O_storage_chunk_bt2_t {
    haddr_t        dset_ohdr_addr; 
    struct H5B2_t *bt2;            
} H5O_storage_chunk_bt2_t;

typedef struct H5O_storage_chunk_t {
    H5D_chunk_index_t             idx_type; 
    haddr_t                       idx_addr; 
    const struct H5D_chunk_ops_t *ops;      
    union {
        H5O_storage_chunk_btree_t       btree;  
        H5O_storage_chunk_bt2_t         btree2; 
        H5O_storage_chunk_earray_t      earray; 
        H5O_storage_chunk_farray_t      farray; 
        H5O_storage_chunk_single_filt_t single; 
    } u;
} H5O_storage_chunk_t;

typedef struct H5O_storage_compact_t {
    bool   dirty; 
    size_t size;  
    void  *buf;   
} H5O_storage_compact_t;

typedef struct H5O_storage_virtual_srcdset_t {
    
    struct H5S_t *virtual_select; 

    
    char         *file_name;              
    char         *dset_name;              
    struct H5S_t *clipped_source_select;  
    struct H5S_t *clipped_virtual_select; 
    struct H5D_t *dset;                   
    bool          dset_exists;            

    
    struct H5S_t *projected_mem_space; 
} H5O_storage_virtual_srcdset_t;

typedef struct H5O_storage_virtual_name_seg_t {
    char                                  *name_segment; 
    struct H5O_storage_virtual_name_seg_t *next;         
} H5O_storage_virtual_name_seg_t;

typedef enum H5O_virtual_space_status_t {
    H5O_VIRTUAL_STATUS_INVALID = 0, 
    H5O_VIRTUAL_STATUS_SEL_BOUNDS,  
    H5O_VIRTUAL_STATUS_USER,        
    H5O_VIRTUAL_STATUS_CORRECT      
} H5O_virtual_space_status_t;

typedef struct H5O_storage_virtual_ent_t {
    
    H5O_storage_virtual_srcdset_t source_dset;      
    char                         *source_file_name; 
    size_t                        source_file_orig; 
    char                         *source_dset_name; 
    size_t                        source_dset_orig; 
    struct H5S_t                 *source_select;    

    
    H5O_storage_virtual_srcdset_t *sub_dset;        
    size_t                         sub_dset_nalloc; 
    size_t sub_dset_nused; 
    size_t sub_dset_io_start; 
    size_t sub_dset_io_end; 
    H5O_storage_virtual_name_seg_t *parsed_source_file_name; 
    size_t psfn_static_strlen; 
    size_t psfn_nsubs;         
    H5O_storage_virtual_name_seg_t *parsed_source_dset_name; 
    size_t  psdn_static_strlen;   
    size_t  psdn_nsubs;           
    int     unlim_dim_source;     
    int     unlim_dim_virtual;    
    hsize_t unlim_extent_source;  
    hsize_t unlim_extent_virtual; 
    hsize_t clip_size_virtual;    
    hsize_t clip_size_source; 
    H5O_virtual_space_status_t source_space_status;  
    H5O_virtual_space_status_t virtual_space_status; 
    UT_hash_handle hh_source_file; 
    UT_hash_handle hh_source_dset; 
} H5O_storage_virtual_ent_t;

typedef struct H5O_storage_virtual_t {
    
    H5HG_t serial_list_hobjid; 

    
    size_t                     list_nused; 
    H5O_storage_virtual_ent_t *list;       

    
    size_t list_nalloc; 
    hsize_t
        min_dims[H5S_MAX_RANK]; 
    H5D_vds_view_t
            view; 
    hsize_t printf_gap; 
    hid_t source_fapl;  
    hid_t source_dapl;  
    bool  init;         
    H5O_storage_virtual_ent_t
        *source_file_hash_table; 
    H5O_storage_virtual_ent_t
        *source_dset_hash_table; 
    H5RT_t *tree;
    size_t  not_in_tree_nused;  
    size_t  not_in_tree_nalloc; 
    H5O_storage_virtual_ent_t *
        *not_in_tree_list; 
} H5O_storage_virtual_t;

typedef struct H5O_storage_t {
    H5D_layout_t type; 
    union {
        H5O_storage_contig_t  contig;  
        H5O_storage_chunk_t   chunk;   
        H5O_storage_compact_t compact; 
        H5O_storage_virtual_t virt;    
    } u;
} H5O_storage_t;

typedef struct H5O_layout_chunk_farray_t {
    
    struct {
        uint8_t max_dblk_page_nelmts_bits; 
    } cparam;
} H5O_layout_chunk_farray_t;

typedef struct H5O_layout_chunk_earray_t {
    
    struct {
        uint8_t max_nelmts_bits; 
        uint8_t idx_blk_elmts;   
        uint8_t data_blk_min_elmts;        
        uint8_t sup_blk_min_data_ptrs;     
        uint8_t max_dblk_page_nelmts_bits; 
    } cparam;

    unsigned unlim_dim;                                 
    hsize_t  swizzled_dim[H5O_LAYOUT_NDIMS];            
    hsize_t  swizzled_down_chunks[H5O_LAYOUT_NDIMS];    
    hsize_t swizzled_max_down_chunks[H5O_LAYOUT_NDIMS]; 
} H5O_layout_chunk_earray_t;

typedef struct H5O_layout_chunk_bt2_t {
    
    struct {
        uint32_t node_size;     
        uint8_t  split_percent; 
        uint8_t  merge_percent; 
    } cparam;
} H5O_layout_chunk_bt2_t;

typedef struct H5O_layout_chunk_t {
    H5D_chunk_index_t idx_type;                      
    uint8_t           flags;                         
    unsigned          ndims;                         
    hsize_t           dim[H5O_LAYOUT_NDIMS];         
    unsigned          enc_bytes_per_dim;             
    hsize_t           size;                          
    hsize_t           nchunks;                       
    hsize_t           max_nchunks;                   
    hsize_t           chunks[H5O_LAYOUT_NDIMS];      
    hsize_t           max_chunks[H5O_LAYOUT_NDIMS];  
    hsize_t           down_chunks[H5O_LAYOUT_NDIMS]; 
    hsize_t           max_down_chunks[H5O_LAYOUT_NDIMS]; 
    union {
        H5O_layout_chunk_farray_t farray; 
        H5O_layout_chunk_earray_t earray; 
        H5O_layout_chunk_bt2_t    btree2; 
    } u;
} H5O_layout_chunk_t;

typedef struct H5O_layout_t {
    H5D_layout_t                   type;    
    unsigned                       version; 
    const struct H5D_layout_ops_t *ops;     
    union {
        H5O_layout_chunk_t chunk; 
    } u;
    H5O_storage_t storage; 
} H5O_layout_t;

#ifdef H5O_ENABLE_BOGUS

#define H5O_BOGUS_VALUE 0xdeadbeef
typedef struct H5O_bogus_t {
    H5O_shared_t sh_loc; 
    unsigned     u;      
} H5O_bogus_t;
#endif 

typedef struct H5O_ginfo_t {
    
    uint32_t lheap_size_hint; 

    

    
    bool     store_link_phase_change; 
    uint16_t max_compact;             
    uint16_t min_dense;               

    
    bool     store_est_entry_info; 
    uint16_t est_num_entries;      
    uint16_t est_name_len;         
} H5O_ginfo_t;

#define H5O_PLINE_VERSION_1 1

#define H5O_PLINE_VERSION_2 2

#define H5O_PLINE_VERSION_LATEST H5O_PLINE_VERSION_2

typedef struct H5O_pline_t {
    H5O_shared_t sh_loc; 

    unsigned           version; 
    size_t             nalloc;  
    size_t             nused;   
    H5Z_filter_info_t *filter;  
} H5O_pline_t;

typedef struct H5O_name_t {
    char *s; 
} H5O_name_t;

typedef struct H5O_shmesg_table_t {
    haddr_t  addr;     
    unsigned version;  
    unsigned nindexes; 
} H5O_shmesg_table_t;

typedef struct H5O_cont_t {
    haddr_t addr; 
    size_t  size; 

    
    unsigned chunkno; 
} H5O_cont_t;

typedef struct H5O_stab_t {
    haddr_t btree_addr; 
    haddr_t heap_addr;  
} H5O_stab_t;

typedef struct H5O_btreek_t {
    unsigned btree_k[H5B_NUM_BTREE_ID]; 
    unsigned sym_leaf_k;                
} H5O_btreek_t;

typedef struct H5O_drvinfo_t {
    
    H5AC_info_t cache_info;
    char        name[9]; 
    size_t      len;     
    uint8_t    *buf;     
} H5O_drvinfo_t;

typedef struct H5O_ainfo_t {
    
    bool              track_corder; 
    bool              index_corder; 
    H5O_msg_crt_idx_t max_crt_idx;  
    haddr_t
        corder_bt2_addr; 

    
    hsize_t nattrs;        
    haddr_t fheap_addr;    
    haddr_t name_bt2_addr; 
} H5O_ainfo_t;

typedef uint32_t H5O_refcount_t; 

typedef unsigned H5O_unknown_t; 

#define H5O_INVALID_VERSION 256

#define H5O_FSINFO_VERSION_0 0

#define H5O_FSINFO_VERSION_1      1
#define H5O_FSINFO_VERSION_LATEST H5O_FSINFO_VERSION_1

typedef struct H5O_fsinfo_t {
    unsigned              version;          
    H5F_fspace_strategy_t strategy;         
    bool                  persist;          
    hsize_t               threshold;        
    hsize_t               page_size;        
    size_t                pgend_meta_thres; 
    haddr_t eoa_pre_fsm_fsalloc; 
    haddr_t fs_addr[H5F_MEM_PAGE_NTYPES - 1]; 
                                              
    bool mapped;                              
                                              
} H5O_fsinfo_t;

typedef struct H5O_mdci_t {
    haddr_t addr; 
    hsize_t size; 
} H5O_mdci_t;

typedef herr_t (*H5O_operator_t)(const void *mesg , unsigned idx, void *operator_data );

typedef herr_t (*H5O_lib_operator_t)(H5O_t *oh, H5O_mesg_t *mesg , unsigned sequence,
                                     void *operator_data );

typedef enum H5O_mesg_operator_type_t {
    H5O_MESG_OP_APP, 
    H5O_MESG_OP_LIB  
} H5O_mesg_operator_type_t;

#define H5O_MODIFY_CONDENSE 0x01
#define H5O_MODIFY          0x02

typedef struct {
    H5O_mesg_operator_type_t op_type;
    union {
        H5O_operator_t     app_op; 
        H5O_lib_operator_t lib_op; 
    } u;
} H5O_mesg_operator_t;

typedef struct {
    H5O_type_t obj_type; 
    void      *crt_info; 
    void      *new_obj;  
} H5O_obj_create_t;

struct H5P_genplist_t;

H5_DLL herr_t H5O_init(void);
H5_DLL herr_t H5O_create(H5F_t *f, size_t size_hint, size_t initial_rc, hid_t ocpl_id,
                         H5O_loc_t *loc );
H5_DLL H5O_t *H5O_create_ohdr(H5F_t *f, hid_t ocpl_id);
H5_DLL herr_t H5O_apply_ohdr(H5F_t *f, H5O_t *oh, hid_t ocpl_id, size_t size_hint, size_t initial_rc,
                             H5O_loc_t *loc_out);
H5_DLL herr_t H5O_open(H5O_loc_t *loc);
H5_DLL void  *H5O_open_by_loc(const H5G_loc_t *obj_loc, H5I_type_t *opened_type );
H5_DLL herr_t H5O_close(H5O_loc_t *loc, bool *file_closed );
H5_DLL int    H5O_link(const H5O_loc_t *loc, int adjust);
H5_DLL H5O_t *H5O_protect(const H5O_loc_t *loc, unsigned prot_flags, bool pin_all_chunks);
H5_DLL H5O_t *H5O_pin(const H5O_loc_t *loc);
H5_DLL herr_t H5O_unpin(H5O_t *oh);
H5_DLL herr_t H5O_dec_rc_by_loc(const H5O_loc_t *loc);
H5_DLL herr_t H5O_unprotect(const H5O_loc_t *loc, H5O_t *oh, unsigned oh_flags);
H5_DLL herr_t H5O_touch(const H5O_loc_t *loc, bool force);
H5_DLL herr_t H5O_touch_oh(H5F_t *f, H5O_t *oh, bool force);
#ifdef H5O_ENABLE_BOGUS
H5_DLL herr_t H5O_bogus_oh(H5F_t *f, H5O_t *oh, unsigned mesg_id, unsigned mesg_flags);
#endif 
H5_DLL herr_t  H5O_delete(H5F_t *f, haddr_t addr);
H5_DLL herr_t  H5O_get_hdr_info(const H5O_loc_t *oloc, H5O_hdr_info_t *hdr);
H5_DLL herr_t  H5O_get_info(const H5O_loc_t *oloc, H5O_info2_t *oinfo, unsigned fields);
H5_DLL herr_t  H5O_get_native_info(const H5O_loc_t *oloc, H5O_native_info_t *oinfo, unsigned fields);
H5_DLL herr_t  H5O_obj_type(const H5O_loc_t *loc, H5O_type_t *obj_type);
H5_DLL herr_t  H5O_get_create_plist(const H5O_loc_t *loc, struct H5P_genplist_t *oc_plist);
H5_DLL void   *H5O_open_name(const H5G_loc_t *loc, const char *name, H5I_type_t *opened_type );
H5_DLL herr_t  H5O_get_nlinks(const H5O_loc_t *loc, hsize_t *nlinks);
H5_DLL void   *H5O_obj_create(H5F_t *f, H5O_type_t obj_type, void *crt_info, H5G_loc_t *obj_loc);
H5_DLL haddr_t H5O_get_oh_addr(const H5O_t *oh);
H5_DLL uint8_t H5O_get_oh_flags(const H5O_t *oh);
H5_DLL time_t  H5O_get_oh_mtime(const H5O_t *oh);
H5_DLL uint8_t H5O_get_oh_version(const H5O_t *oh);
H5_DLL herr_t  H5O_get_rc_and_type(const H5O_loc_t *oloc, unsigned *rc, H5O_type_t *otype);
H5_DLL H5AC_proxy_entry_t *H5O_get_proxy(const H5O_t *oh);
H5_DLL bool                H5O_has_chksum(const H5O_t *oh);
H5_DLL herr_t              H5O_get_version_bound(H5F_libver_t bound, uint8_t *version);

H5_DLL herr_t H5O_msg_create(const H5O_loc_t *loc, unsigned type_id, unsigned mesg_flags,
                             unsigned update_flags, void *mesg);
H5_DLL herr_t H5O_msg_append_oh(H5F_t *f, H5O_t *oh, unsigned type_id, unsigned mesg_flags,
                                unsigned update_flags, void *mesg);
H5_DLL herr_t H5O_msg_write(const H5O_loc_t *loc, unsigned type_id, unsigned mesg_flags,
                            unsigned update_flags, void *mesg);
H5_DLL herr_t H5O_msg_write_oh(H5F_t *f, H5O_t *oh, unsigned type_id, unsigned mesg_flags,
                               unsigned update_flags, void *mesg);
H5_DLL herr_t H5O_msg_flush(H5F_t *f, H5O_t *oh, H5O_mesg_t *mesg);
H5_DLL void  *H5O_msg_read(const H5O_loc_t *loc, unsigned type_id, void *mesg);
H5_DLL void  *H5O_msg_read_oh(H5F_t *f, H5O_t *oh, unsigned type_id, void *mesg);
H5_DLL herr_t H5O_msg_reset(unsigned type_id, void *native);
H5_DLL void  *H5O_msg_free(unsigned type_id, void *mesg);
H5_DLL void  *H5O_msg_free_real(const H5O_msg_class_t *type, void *mesg);
H5_DLL void  *H5O_msg_copy(unsigned type_id, const void *mesg, void *dst);
H5_DLL int    H5O_msg_count(const H5O_loc_t *loc, unsigned type_id);
H5_DLL htri_t H5O_msg_exists(const H5O_loc_t *loc, unsigned type_id);
H5_DLL htri_t H5O_msg_exists_oh(const H5O_t *oh, unsigned type_id);
H5_DLL herr_t H5O_msg_remove(const H5O_loc_t *loc, unsigned type_id, int sequence, bool adj_link);
H5_DLL herr_t H5O_msg_remove_op(const H5O_loc_t *loc, unsigned type_id, int sequence, H5O_operator_t op,
                                void *op_data, bool adj_link);
H5_DLL herr_t H5O_msg_iterate(const H5O_loc_t *loc, unsigned type_id, const H5O_mesg_operator_t *op,
                              void *op_data);
H5_DLL size_t H5O_msg_raw_size(const H5F_t *f, unsigned type_id, bool disable_shared, const void *mesg);
H5_DLL size_t H5O_msg_size_f(const H5F_t *f, hid_t ocpl_id, unsigned type_id, const void *mesg,
                             size_t extra_raw);
H5_DLL size_t H5O_msg_size_oh(const H5F_t *f, const H5O_t *oh, unsigned type_id, const void *mesg,
                              size_t extra_raw);
H5_DLL htri_t H5O_msg_is_shared(unsigned type_id, const void *mesg);
H5_DLL htri_t H5O_msg_can_share(unsigned type_id, const void *mesg);
H5_DLL htri_t H5O_msg_can_share_in_ohdr(unsigned type_id);
H5_DLL herr_t H5O_msg_set_share(unsigned type_id, const H5O_shared_t *share, void *mesg);
H5_DLL herr_t H5O_msg_reset_share(unsigned type_id, void *mesg);
H5_DLL herr_t H5O_msg_get_crt_index(unsigned type_id, const void *mesg, H5O_msg_crt_idx_t *crt_idx);
H5_DLL herr_t H5O_msg_encode(H5F_t *f, unsigned type_id, bool disable_shared, unsigned char *buf,
                             const void *obj);
H5_DLL void  *H5O_msg_decode(H5F_t *f, H5O_t *open_oh, unsigned type_id, size_t buf_size,
                             const unsigned char *buf);
H5_DLL herr_t H5O_msg_delete(H5F_t *f, H5O_t *open_oh, unsigned type_id, void *mesg);
H5_DLL herr_t H5O_msg_get_flags(const H5O_loc_t *loc, unsigned type_id, uint8_t *flags);

H5_DLL herr_t H5O_flush(H5O_loc_t *oloc, hid_t obj_id);
H5_DLL herr_t H5O_flush_common(H5O_loc_t *oloc, hid_t obj_id);
H5_DLL herr_t H5O_refresh_metadata(H5O_loc_t *oloc, hid_t oid);
H5_DLL herr_t H5O_refresh_metadata_reopen(hid_t oid, hid_t apl_id, H5G_loc_t *obj_loc,
                                          H5VL_connector_t *connector, bool start_swmr);

H5_DLL herr_t H5O_copy_header_map(const H5O_loc_t *oloc_src, H5O_loc_t *oloc_dst ,
                                  H5O_copy_t *cpy_info, bool inc_depth, H5O_type_t *obj_type, void **udata);
H5_DLL herr_t H5O_copy_expand_ref(H5F_t *file_src, const H5T_t *dt_src, void *buf_src, size_t nbytes_src,
                                  H5F_t *file_dst, void *buf_dst, H5O_copy_t *cpy_info);

H5_DLL herr_t H5O_debug_id(unsigned type_id, H5F_t *f, const void *mesg, FILE *stream, int indent,
                           int fwidth);
H5_DLL herr_t H5O_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth);

H5_DLL herr_t     H5O_loc_reset(H5O_loc_t *loc);
H5_DLL herr_t     H5O_loc_copy(H5O_loc_t *dst, H5O_loc_t *src, H5_copy_depth_t depth);
H5_DLL herr_t     H5O_loc_copy_shallow(H5O_loc_t *dst, H5O_loc_t *src);
H5_DLL herr_t     H5O_loc_copy_deep(H5O_loc_t *dst, const H5O_loc_t *src);
H5_DLL herr_t     H5O_loc_hold_file(H5O_loc_t *loc);
H5_DLL herr_t     H5O_loc_free(H5O_loc_t *loc);
H5_DLL H5O_loc_t *H5O_get_loc(hid_t id);

H5_DLL herr_t H5O_efl_total_size(const H5O_efl_t *efl, hsize_t *size);

H5_DLL herr_t H5O_fsinfo_set_version(H5F_libver_t low, H5F_libver_t high, H5O_fsinfo_t *fsinfo);
H5_DLL herr_t H5O_fsinfo_check_version(H5F_libver_t high, H5O_fsinfo_t *fsinfo);

H5_DLL herr_t H5O_fill_reset_dyn(H5O_fill_t *fill);
H5_DLL herr_t H5O_fill_convert(H5O_fill_t *fill, H5T_t *type, bool *fill_changed);
H5_DLL herr_t H5O_fill_set_version(H5F_t *f, H5O_fill_t *fill);

H5_DLL herr_t H5O_link_delete(H5F_t *f, H5O_t *open_oh, void *_mesg);

H5_DLL herr_t H5O_pline_set_version(H5F_t *f, H5O_pline_t *pline);

H5_DLL herr_t H5O_set_shared(H5O_shared_t *dst, const H5O_shared_t *src);

#endif 
