## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.align = 'center',
  fig.width = 9,
  fig.height = 6,
  warning = F,
  message = F
)

## ----warning = FALSE, results=FALSE, force=TRUE, include=FALSE----------------
if (!requireNamespace("bayesmsm", quietly = TRUE)) {
  stop("The package 'bayesmsm' is required to run this vignette. Please install it manually using:
       devtools::install_github('Kuan-Liu-Lab/bayesmsm')")
} else {
  library(bayesmsm)
}

## -----------------------------------------------------------------------------
# 1) Define coefficient lists for 2 visits
amodel <- list(
  # Visit 1: logit P(A1=1) = -0.3 + 0.4*L1_1 - 0.2*L2_1
  c("(Intercept)" = -0.3, "L1_1" = 0.4, "L2_1" = -0.2),
  # Visit 2: logit P(A2=1) = -0.1 + 0.3*L1_2 - 0.1*L2_2 + 0.5*A_prev
  c("(Intercept)" = -0.1, "L1_2" = 0.3, "L2_2" = -0.1, "A_prev" = 0.5)
)

# 2) Define binary outcome model: logistic on treatments and last covariates
ymodel <- c(
  "(Intercept)" = -0.8,
  "A1"          = 0.2,
  "A2"          = 0.4,
  "L1_2"        = 0.3,
  "L2_2"        = -0.3
)

# 3) Load package and simulate data without censoring
simdat <- simData(
  n                = 100,
  n_visits         = 2,
  covariate_counts = c(2, 2),
  amodel           = amodel,
  ymodel           = ymodel,
  y_type           = "binary",
  right_censor     = FALSE,
  seed             = 123
)

# 4) Inspect first rows
head(simdat)

## -----------------------------------------------------------------------------
weights <- bayesweight(
  trtmodel.list = list(
    A1 ~ L1_1 + L2_1,
    A2 ~ L1_2 + L2_2 + A1),
  data = simdat,
  n.chains = 1,
  n.iter = 200,
  n.burnin = 100,
  n.thin = 1,
  seed = 890123,
  parallel = FALSE)

summary(weights)

## -----------------------------------------------------------------------------
cat(weights$model_string)

## -----------------------------------------------------------------------------
model <- bayesmsm(ymodel = Y ~ A1 + A2,
  nvisit = 2,
  reference = c(rep(0,2)),
  comparator = c(rep(1,2)),
  family = "binomial",
  data = simdat,
  wmean = weights$weights,
  nboot = 50,
  optim_method = "BFGS",
  parallel = TRUE,
  seed = 890123,
  ncore = 2)
str(model)

## -----------------------------------------------------------------------------
summary_bayesmsm(model)

## -----------------------------------------------------------------------------
plot_ATE(model)

## -----------------------------------------------------------------------------
plot_APO(model, "effect_reference")
plot_APO(model, "effect_comparator")

## -----------------------------------------------------------------------------
plot_est_box(model)

