﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 * <p>Contains the parameters for DeleteNetworkInterfacePermission.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteNetworkInterfacePermissionRequest">AWS
 * API Reference</a></p>
 */
class DeleteNetworkInterfacePermissionRequest : public EC2Request {
 public:
  AWS_EC2_API DeleteNetworkInterfacePermissionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteNetworkInterfacePermission"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ID of the network interface permission.</p>
   */
  inline const Aws::String& GetNetworkInterfacePermissionId() const { return m_networkInterfacePermissionId; }
  inline bool NetworkInterfacePermissionIdHasBeenSet() const { return m_networkInterfacePermissionIdHasBeenSet; }
  template <typename NetworkInterfacePermissionIdT = Aws::String>
  void SetNetworkInterfacePermissionId(NetworkInterfacePermissionIdT&& value) {
    m_networkInterfacePermissionIdHasBeenSet = true;
    m_networkInterfacePermissionId = std::forward<NetworkInterfacePermissionIdT>(value);
  }
  template <typename NetworkInterfacePermissionIdT = Aws::String>
  DeleteNetworkInterfacePermissionRequest& WithNetworkInterfacePermissionId(NetworkInterfacePermissionIdT&& value) {
    SetNetworkInterfacePermissionId(std::forward<NetworkInterfacePermissionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify <code>true</code> to remove the permission even if the network
   * interface is attached to an instance.</p>
   */
  inline bool GetForce() const { return m_force; }
  inline bool ForceHasBeenSet() const { return m_forceHasBeenSet; }
  inline void SetForce(bool value) {
    m_forceHasBeenSet = true;
    m_force = value;
  }
  inline DeleteNetworkInterfacePermissionRequest& WithForce(bool value) {
    SetForce(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline DeleteNetworkInterfacePermissionRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_networkInterfacePermissionId;
  bool m_networkInterfacePermissionIdHasBeenSet = false;

  bool m_force{false};
  bool m_forceHasBeenSet = false;

  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
