/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.shuffle.checksum;

import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.TimeUnit;
import java.util.zip.Adler32;
import java.util.zip.CRC32;
import java.util.zip.CRC32C;
import java.util.zip.CheckedInputStream;
import java.util.zip.Checksum;
import org.apache.spark.annotation.Private;
import org.apache.spark.internal.LogKey;
import org.apache.spark.internal.LogKeys;
import org.apache.spark.internal.MDC;
import org.apache.spark.internal.SparkLogger;
import org.apache.spark.internal.SparkLoggerFactory;
import org.apache.spark.network.buffer.ManagedBuffer;
import org.apache.spark.network.shuffle.checksum.Cause;

@Private
public class ShuffleChecksumHelper {
    private static final SparkLogger logger = SparkLoggerFactory.getLogger(ShuffleChecksumHelper.class);
    public static final int CHECKSUM_CALCULATION_BUFFER = 8192;
    public static final Checksum[] EMPTY_CHECKSUM = new Checksum[0];
    public static final long[] EMPTY_CHECKSUM_VALUE = new long[0];

    public static Checksum[] createPartitionChecksums(int numPartitions, String algorithm) {
        return ShuffleChecksumHelper.getChecksumsByAlgorithm(numPartitions, algorithm);
    }

    private static Checksum[] getChecksumsByAlgorithm(int num, String algorithm) {
        Checksum[] checksums;
        switch (algorithm) {
            case "ADLER32": {
                checksums = new Adler32[num];
                for (int i = 0; i < num; ++i) {
                    checksums[i] = new Adler32();
                }
                break;
            }
            case "CRC32": {
                checksums = new CRC32[num];
                for (int i = 0; i < num; ++i) {
                    checksums[i] = new CRC32();
                }
                break;
            }
            case "CRC32C": {
                checksums = new CRC32C[num];
                for (int i = 0; i < num; ++i) {
                    checksums[i] = new CRC32C();
                }
                break;
            }
            default: {
                throw new UnsupportedOperationException("Unsupported shuffle checksum algorithm: " + algorithm);
            }
        }
        return checksums;
    }

    public static Checksum getChecksumByAlgorithm(String algorithm) {
        return ShuffleChecksumHelper.getChecksumsByAlgorithm(1, algorithm)[0];
    }

    public static String getChecksumFileName(String blockName, String algorithm) {
        return String.format("%s.%s", blockName, algorithm);
    }

    private static long readChecksumByReduceId(File checksumFile, int reduceId) throws IOException {
        try (DataInputStream in = new DataInputStream(new FileInputStream(checksumFile));){
            in.skipNBytes((long)reduceId * 8L);
            long l = in.readLong();
            return l;
        }
    }

    private static long calculateChecksumForPartition(ManagedBuffer partitionData, Checksum checksumAlgo) throws IOException {
        InputStream in = partitionData.createInputStream();
        byte[] buffer = new byte[8192];
        try (CheckedInputStream checksumIn = new CheckedInputStream(in, checksumAlgo);){
            while (checksumIn.read(buffer, 0, 8192) != -1) {
            }
            long l = checksumAlgo.getValue();
            return l;
        }
    }

    public static Cause diagnoseCorruption(String algorithm, File checksumFile, int reduceId, ManagedBuffer partitionData, long checksumByReader) {
        Cause cause;
        long duration = -1L;
        long checksumByWriter = -1L;
        long checksumByReCalculation = -1L;
        try {
            long diagnoseStartNs = System.nanoTime();
            Checksum checksumAlgo = ShuffleChecksumHelper.getChecksumByAlgorithm(algorithm);
            checksumByWriter = ShuffleChecksumHelper.readChecksumByReduceId(checksumFile, reduceId);
            checksumByReCalculation = ShuffleChecksumHelper.calculateChecksumForPartition(partitionData, checksumAlgo);
            duration = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - diagnoseStartNs);
            cause = checksumByWriter != checksumByReCalculation ? Cause.DISK_ISSUE : (checksumByWriter != checksumByReader ? Cause.NETWORK_ISSUE : Cause.CHECKSUM_VERIFY_PASS);
        }
        catch (UnsupportedOperationException e) {
            cause = Cause.UNSUPPORTED_CHECKSUM_ALGORITHM;
        }
        catch (FileNotFoundException e) {
            logger.warn("Checksum file {} doesn't exit", new MDC[]{MDC.of((LogKey)LogKeys.PATH, (Object)checksumFile.getName())});
            cause = Cause.UNKNOWN_ISSUE;
        }
        catch (Exception e) {
            logger.warn("Unable to diagnose shuffle block corruption", (Throwable)e);
            cause = Cause.UNKNOWN_ISSUE;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Shuffle corruption diagnosis took {} ms, checksum file {}, cause {}, checksumByReader {}, checksumByWriter {}, checksumByReCalculation {}", new Object[]{duration, checksumFile.getAbsolutePath(), cause, checksumByReader, checksumByWriter, checksumByReCalculation});
        } else {
            logger.info("Shuffle corruption diagnosis took {} ms, checksum file {}, cause {}", new MDC[]{MDC.of((LogKey)LogKeys.TIME, (Object)duration), MDC.of((LogKey)LogKeys.PATH, (Object)checksumFile.getAbsolutePath()), MDC.of((LogKey)LogKeys.REASON, (Object)((Object)cause))});
        }
        return cause;
    }
}

