/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.conf;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.time.Duration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;
import org.apache.hadoop.hdds.conf.Config;
import org.apache.hadoop.hdds.conf.ConfigGroup;
import org.apache.hadoop.hdds.conf.ConfigType;
import org.apache.hadoop.hdds.conf.ConfigurationException;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.ConfigurationTarget;
import org.apache.hadoop.hdds.conf.PostConstruct;

public final class ConfigurationReflectionUtil {
    private ConfigurationReflectionUtil() {
    }

    public static <T> Map<String, Field> mapReconfigurableProperties(Class<T> configurationClass) {
        String prefix = ConfigurationReflectionUtil.getPrefix(configurationClass);
        Map<String, Field> props = ConfigurationReflectionUtil.mapReconfigurableProperties(configurationClass, prefix);
        for (Class<T> superClass = configurationClass.getSuperclass(); superClass != null; superClass = superClass.getSuperclass()) {
            props.putAll(ConfigurationReflectionUtil.mapReconfigurableProperties(superClass, prefix));
        }
        return props;
    }

    private static <T> Map<String, Field> mapReconfigurableProperties(Class<T> configurationClass, String prefix) {
        HashMap<String, Field> props = new HashMap<String, Field>();
        for (Field field : configurationClass.getDeclaredFields()) {
            Config configAnnotation;
            if (!field.isAnnotationPresent(Config.class) || !(configAnnotation = field.getAnnotation(Config.class)).reconfigurable()) continue;
            ConfigurationReflectionUtil.checkNotFinal(configurationClass, field);
            props.put(ConfigurationReflectionUtil.getFullKey(prefix, configAnnotation), field);
        }
        return props;
    }

    public static <T> void injectConfiguration(ConfigurationSource configuration, Class<T> configurationClass, T configObject, boolean reconfiguration) {
        String prefix = ConfigurationReflectionUtil.getPrefix(configurationClass);
        ConfigurationReflectionUtil.injectConfigurationToObject(configuration, configurationClass, configObject, prefix, reconfiguration);
        for (Class<T> superClass = configurationClass.getSuperclass(); superClass != null; superClass = superClass.getSuperclass()) {
            ConfigurationReflectionUtil.injectConfigurationToObject(configuration, superClass, configObject, prefix, reconfiguration);
        }
    }

    private static <T> void injectConfigurationToObject(ConfigurationSource from, Class<T> configurationClass, T configuration, String prefix, boolean reconfiguration) {
        for (Field field : configurationClass.getDeclaredFields()) {
            if (!field.isAnnotationPresent(Config.class)) continue;
            ConfigurationReflectionUtil.checkNotFinal(configurationClass, field);
            Config configAnnotation = field.getAnnotation(Config.class);
            if (reconfiguration && !configAnnotation.reconfigurable()) continue;
            String key = ConfigurationReflectionUtil.getFullKey(prefix, configAnnotation);
            String defaultValue = configAnnotation.defaultValue();
            String value = from.get(key, defaultValue);
            ConfigurationReflectionUtil.setField(configurationClass, configuration, field, configAnnotation, key, value);
        }
    }

    public static <T> void reconfigureProperty(T configuration, Field field, String key, String value) {
        Class<?> klass = field.getDeclaringClass();
        if (!field.isAnnotationPresent(Config.class)) {
            throw new ConfigurationException("Not configurable field: " + klass + "." + field.getName());
        }
        Config configAnnotation = field.getAnnotation(Config.class);
        try {
            Object oldValue = ConfigurationReflectionUtil.forcedFieldGet(field, configuration);
            ConfigurationReflectionUtil.setField(klass, configuration, field, configAnnotation, key, value);
            try {
                ConfigurationReflectionUtil.callPostConstruct(configuration);
            }
            catch (Exception e) {
                ConfigurationReflectionUtil.forcedFieldSet(field, configuration, oldValue);
                throw e;
            }
        }
        catch (IllegalAccessException e) {
            throw new ConfigurationException("Failed to inject configuration to " + klass.getSimpleName() + "." + field.getName(), e);
        }
    }

    private static <T> void setField(Class<?> configurationClass, T configuration, Field field, Config configAnnotation, String key, String value) {
        ConfigType type = configAnnotation.type();
        if (type == ConfigType.AUTO) {
            type = ConfigurationReflectionUtil.detectConfigType(field);
        }
        try {
            Object parsed = type.parse(value, configAnnotation, field.getType(), key);
            ConfigurationReflectionUtil.forcedFieldSet(field, configuration, parsed);
        }
        catch (Exception e) {
            throw new ConfigurationException("Failed to inject configuration to " + configurationClass.getSimpleName() + "." + field.getName(), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static <T> void forcedFieldSet(Field field, T object, Object value) throws IllegalAccessException {
        boolean accessChanged = ConfigurationReflectionUtil.setAccessible(field);
        try {
            field.set(object, value);
        }
        finally {
            if (accessChanged) {
                field.setAccessible(false);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static <T> Object forcedFieldGet(Field field, T object) throws IllegalAccessException {
        boolean accessChanged = ConfigurationReflectionUtil.setAccessible(field);
        try {
            Object object2 = field.get(object);
            return object2;
        }
        finally {
            if (accessChanged) {
                field.setAccessible(false);
            }
        }
    }

    private static boolean setAccessible(Field field) {
        if (!field.isAccessible()) {
            field.setAccessible(true);
            return true;
        }
        return false;
    }

    private static ConfigType detectConfigType(Field field) {
        ConfigType type;
        Class<?> parameterType = field.getType();
        if (parameterType == String.class) {
            type = ConfigType.STRING;
        } else if (parameterType == Integer.class || parameterType == Integer.TYPE) {
            type = ConfigType.INT;
        } else if (parameterType == Long.class || parameterType == Long.TYPE) {
            type = ConfigType.LONG;
        } else if (parameterType == Double.class || parameterType == Double.TYPE) {
            type = ConfigType.DOUBLE;
        } else if (parameterType == Boolean.class || parameterType == Boolean.TYPE) {
            type = ConfigType.BOOLEAN;
        } else if (parameterType == Duration.class) {
            type = ConfigType.TIME;
        } else if (parameterType == Class.class) {
            type = ConfigType.CLASS;
        } else {
            throw new ConfigurationException("Unsupported configuration type " + parameterType + " in " + field.getDeclaringClass() + "." + field.getName());
        }
        return type;
    }

    static <T> void callPostConstruct(T configObject) {
        Class<?> configurationClass = configObject.getClass();
        for (Method method : configurationClass.getMethods()) {
            if (!method.isAnnotationPresent(PostConstruct.class)) continue;
            try {
                method.invoke(configObject, new Object[0]);
            }
            catch (IllegalAccessException ex) {
                throw new IllegalArgumentException("@PostConstruct method in " + configurationClass + " is not accessible");
            }
            catch (InvocationTargetException e) {
                if (e.getCause() instanceof RuntimeException) {
                    throw (RuntimeException)e.getCause();
                }
                throw new IllegalArgumentException("@PostConstruct can't be executed on " + configurationClass + " after configObject injection", e);
            }
        }
    }

    public static <T> void updateConfiguration(ConfigurationTarget config, T object) {
        ConfigurationReflectionUtil.updateConfiguration(config, object, ConfigurationReflectionUtil.getPrefix(object.getClass()));
    }

    private static <T> void updateConfiguration(ConfigurationTarget config, T object, String prefix) {
        Class<?> configClass = object.getClass();
        LinkedList classes = new LinkedList();
        classes.addLast(configClass);
        for (Class<?> superclass = configClass.getSuperclass(); superclass != null; superclass = superclass.getSuperclass()) {
            classes.addFirst(superclass);
        }
        for (Class clazz : classes) {
            ConfigurationReflectionUtil.updateConfigurationFromObject(config, clazz, object, prefix);
        }
    }

    private static <T> void updateConfigurationFromObject(ConfigurationTarget config, Class<?> configClass, T configObject, String prefix) {
        for (Field field : configClass.getDeclaredFields()) {
            if (!field.isAnnotationPresent(Config.class)) continue;
            Config configAnnotation = field.getAnnotation(Config.class);
            String fieldLocation = configClass + "." + field.getName();
            String key = ConfigurationReflectionUtil.getFullKey(prefix, configAnnotation);
            ConfigType type = configAnnotation.type();
            if (type == ConfigType.AUTO) {
                type = ConfigurationReflectionUtil.detectConfigType(field);
            }
            try {
                Object value = ConfigurationReflectionUtil.forcedFieldGet(field, configObject);
                if (value == null) continue;
                type.set(config, key, value, configAnnotation);
            }
            catch (IllegalAccessException e) {
                throw new ConfigurationException("Can't inject configuration to " + fieldLocation, e);
            }
        }
    }

    public static Optional<String> getDefaultValue(Class<?> configClass, String fieldName) {
        return ConfigurationReflectionUtil.findFieldConfigAnnotationByName(configClass, fieldName).map(Config::defaultValue);
    }

    public static Optional<String> getKey(Class<?> configClass, String fieldName) {
        ConfigGroup configGroup = ConfigurationReflectionUtil.getConfigGroup(configClass);
        return ConfigurationReflectionUtil.findFieldConfigAnnotationByName(configClass, fieldName).map(config -> ConfigurationReflectionUtil.getFullKey(configGroup, config));
    }

    public static Optional<ConfigType> getType(Class<?> configClass, String fieldName) {
        return ConfigurationReflectionUtil.findFieldConfigAnnotationByName(configClass, fieldName).map(Config::type);
    }

    private static Optional<Config> findFieldConfigAnnotationByName(Class<?> configClass, String fieldName) {
        Class<?> theClass = configClass;
        while (theClass != null) {
            Optional<Config> config = Stream.of(theClass.getDeclaredFields()).filter(f -> f.getName().equals(fieldName)).findFirst().map(f -> f.getAnnotation(Config.class));
            if (config.isPresent()) {
                return config;
            }
            if (!Object.class.equals(theClass = theClass.getSuperclass())) continue;
            theClass = null;
        }
        return Optional.empty();
    }

    private static void checkNotFinal(Class<?> configurationClass, Field field) {
        if ((field.getModifiers() & 0x10) != 0) {
            throw new ConfigurationException(String.format("Trying to set final field %s#%s, probably indicates misplaced @Config annotation", configurationClass.getSimpleName(), field.getName()));
        }
    }

    public static String getFullKey(ConfigGroup configGroup, Config configAnnotation) {
        return ConfigurationReflectionUtil.getFullKey(ConfigurationReflectionUtil.getPrefix(configGroup), configAnnotation);
    }

    private static String getPrefix(Class<?> configurationClass) {
        return ConfigurationReflectionUtil.getPrefix(ConfigurationReflectionUtil.getConfigGroup(configurationClass));
    }

    private static ConfigGroup getConfigGroup(Class<?> configurationClass) {
        return configurationClass.getAnnotation(ConfigGroup.class);
    }

    private static String getPrefix(ConfigGroup configGroup) {
        return configGroup != null && !configGroup.prefix().isEmpty() ? configGroup.prefix() + "." : "";
    }

    private static String getFullKey(String prefix, Config configAnnotation) {
        String key = configAnnotation.key();
        return prefix != null && !prefix.isEmpty() && !key.startsWith(prefix) ? prefix + key : key;
    }
}

