/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.storage;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.EOFException;
import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.client.StandaloneReplicationConfig;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.OzoneClientConfig;
import org.apache.hadoop.hdds.scm.XceiverClientFactory;
import org.apache.hadoop.hdds.scm.XceiverClientSpi;
import org.apache.hadoop.hdds.scm.client.HddsClientUtils;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.storage.BlockExtendedInputStream;
import org.apache.hadoop.hdds.scm.storage.BlockLocationInfo;
import org.apache.hadoop.hdds.scm.storage.ByteReaderStrategy;
import org.apache.hadoop.hdds.scm.storage.ChunkInputStream;
import org.apache.hadoop.hdds.scm.storage.ContainerProtocolCalls;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.apache.hadoop.hdds.security.token.OzoneBlockTokenIdentifier;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.security.token.Token;
import org.apache.ratis.thirdparty.io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockInputStream
extends BlockExtendedInputStream {
    public static final Logger LOG = LoggerFactory.getLogger(BlockInputStream.class);
    private static final List<XceiverClientSpi.Validator> VALIDATORS = ContainerProtocolCalls.toValidatorList((request, response) -> BlockInputStream.validate(response));
    private final BlockID blockID;
    private long length;
    private final BlockLocationInfo blockInfo;
    private final AtomicReference<Pipeline> pipelineRef = new AtomicReference();
    private final AtomicReference<Token<OzoneBlockTokenIdentifier>> tokenRef = new AtomicReference();
    private final boolean verifyChecksum;
    private XceiverClientFactory xceiverClientFactory;
    private XceiverClientSpi xceiverClient;
    private boolean initialized = false;
    private final RetryPolicy retryPolicy;
    private int retries;
    private List<ChunkInputStream> chunkStreams;
    private long[] chunkOffsets = null;
    private int chunkIndex;
    private long blockPosition = 0L;
    private int chunkIndexOfPrevPosition;
    private final Function<BlockID, BlockLocationInfo> refreshFunction;

    public BlockInputStream(BlockLocationInfo blockInfo, Pipeline pipeline, Token<OzoneBlockTokenIdentifier> token, XceiverClientFactory xceiverClientFactory, Function<BlockID, BlockLocationInfo> refreshFunction, OzoneClientConfig config) throws IOException {
        this.blockInfo = blockInfo;
        this.blockID = blockInfo.getBlockID();
        this.length = blockInfo.getLength();
        this.setPipeline(pipeline);
        this.tokenRef.set(token);
        this.verifyChecksum = config.isChecksumVerify();
        this.xceiverClientFactory = xceiverClientFactory;
        this.refreshFunction = refreshFunction;
        this.retryPolicy = HddsClientUtils.createRetryPolicy(config.getMaxReadRetryCount(), TimeUnit.SECONDS.toMillis(config.getReadRetryInterval()));
    }

    public BlockInputStream(BlockID blockId, long blockLen, Pipeline pipeline, Token<OzoneBlockTokenIdentifier> token, XceiverClientFactory xceiverClientFactory, OzoneClientConfig config) throws IOException {
        this(new BlockLocationInfo(new BlockLocationInfo.Builder().setBlockID(blockId).setLength(blockLen)), pipeline, token, xceiverClientFactory, null, config);
    }

    public synchronized void initialize() throws IOException {
        if (this.initialized) {
            return;
        }
        ContainerProtos.BlockData blockData = null;
        List chunks = null;
        Throwable catchEx = null;
        do {
            try {
                blockData = this.getBlockData();
                chunks = blockData.getChunksList();
                if (this.blockInfo == null || !this.blockInfo.isUnderConstruction()) break;
                this.length = blockData.getSize();
                LOG.debug("Updated block length to {} for block {}", (Object)this.length, (Object)this.blockID);
                break;
            }
            catch (SCMSecurityException ex) {
                throw ex;
            }
            catch (StorageContainerException ex) {
                this.refreshBlockInfo((IOException)((Object)ex));
                catchEx = ex;
            }
            catch (IOException ex) {
                LOG.debug("Retry to get chunk info fail", (Throwable)ex);
                if (this.isConnectivityIssue(ex)) {
                    this.refreshBlockInfo(ex);
                }
                catchEx = ex;
            }
        } while (this.shouldRetryRead((IOException)catchEx));
        if (chunks == null) {
            throw catchEx;
        }
        this.retries = 0;
        if (!chunks.isEmpty()) {
            this.chunkOffsets = new long[chunks.size()];
            long tempOffset = 0L;
            this.chunkStreams = new ArrayList<ChunkInputStream>(chunks.size());
            for (int i = 0; i < chunks.size(); ++i) {
                this.addStream((ContainerProtos.ChunkInfo)chunks.get(i));
                this.chunkOffsets[i] = tempOffset;
                tempOffset += ((ContainerProtos.ChunkInfo)chunks.get(i)).getLen();
            }
            this.initialized = true;
            this.chunkIndex = 0;
            if (this.blockPosition > 0L) {
                this.seek(this.blockPosition);
            }
        }
    }

    private boolean isConnectivityIssue(IOException ex) {
        return Status.fromThrowable((Throwable)ex).getCode() == Status.UNAVAILABLE.getCode();
    }

    private void refreshBlockInfo(IOException cause) throws IOException {
        LOG.info("Attempting to update pipeline and block token for block {} from pipeline {}: {}", new Object[]{this.blockID, this.pipelineRef.get().getId(), cause.getMessage()});
        if (this.refreshFunction != null) {
            LOG.debug("Re-fetching pipeline and block token for block {}", (Object)this.blockID);
            BlockLocationInfo blockLocationInfo = this.refreshFunction.apply(this.blockID);
            if (blockLocationInfo == null) {
                LOG.warn("No new block location info for block {}", (Object)this.blockID);
            } else {
                this.setPipeline(blockLocationInfo.getPipeline());
                LOG.info("New pipeline for block {}: {}", (Object)this.blockID, (Object)blockLocationInfo.getPipeline());
                this.tokenRef.set((Token<OzoneBlockTokenIdentifier>)blockLocationInfo.getToken());
                if (blockLocationInfo.getToken() != null) {
                    OzoneBlockTokenIdentifier tokenId = new OzoneBlockTokenIdentifier();
                    tokenId.readFromByteArray(this.tokenRef.get().getIdentifier());
                    LOG.info("A new token is added for block {}. Expiry: {}", (Object)this.blockID, (Object)Instant.ofEpochMilli(tokenId.getExpiryDate()));
                }
            }
        } else {
            throw cause;
        }
    }

    protected ContainerProtos.BlockData getBlockData() throws IOException {
        this.acquireClient();
        try {
            ContainerProtos.BlockData blockData = this.getBlockDataUsingClient();
            return blockData;
        }
        finally {
            this.releaseClient();
        }
    }

    protected ContainerProtos.BlockData getBlockDataUsingClient() throws IOException {
        Pipeline pipeline = this.pipelineRef.get();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Initializing BlockInputStream for get key to access block {}", (Object)this.blockID);
        }
        ContainerProtos.GetBlockResponseProto response = ContainerProtocolCalls.getBlock((XceiverClientSpi)this.xceiverClient, VALIDATORS, (BlockID)this.blockID, this.tokenRef.get(), (Map)pipeline.getReplicaIndexes());
        return response.getBlockData();
    }

    private void setPipeline(Pipeline pipeline) throws IOException {
        if (pipeline == null) {
            return;
        }
        long replicaIndexes = pipeline.getNodes().stream().mapToInt(arg_0 -> ((Pipeline)pipeline).getReplicaIndex(arg_0)).distinct().count();
        if (replicaIndexes > 1L) {
            throw new IOException(String.format("Pipeline: %s has nodes containing different replica indexes.", pipeline));
        }
        boolean okForRead = pipeline.getType() == HddsProtos.ReplicationType.STAND_ALONE || pipeline.getType() == HddsProtos.ReplicationType.EC;
        Pipeline readPipeline = okForRead ? pipeline : Pipeline.newBuilder((Pipeline)pipeline).setReplicationConfig((ReplicationConfig)StandaloneReplicationConfig.getInstance((HddsProtos.ReplicationFactor)ReplicationConfig.getLegacyFactor((ReplicationConfig)pipeline.getReplicationConfig()))).build();
        this.pipelineRef.set(readPipeline);
    }

    private static void validate(ContainerProtos.ContainerCommandResponseProto response) throws IOException {
        if (!response.hasGetBlock()) {
            throw new IllegalArgumentException("Not GetBlock: response=" + response);
        }
        ContainerProtos.GetBlockResponseProto b = response.getGetBlock();
        long blockLength = b.getBlockData().getSize();
        List chunks = b.getBlockData().getChunksList();
        for (int i = 0; i < chunks.size(); ++i) {
            ContainerProtos.ChunkInfo c = (ContainerProtos.ChunkInfo)chunks.get(i);
            if (c.getLen() <= 0L && i == chunks.size() - 1 && c.getOffset() == blockLength) {
                ContainerProtos.DatanodeBlockID blockID = b.getBlockData().getBlockID();
                LOG.warn("The last chunk is empty for container/block {}/{} with an offset of the block length. Likely due to HDDS-10682. This is safe to ignore.", (Object)blockID.getContainerID(), (Object)blockID.getLocalID());
                continue;
            }
            if (c.getLen() > 0L) continue;
            throw new IOException("Failed to get chunkInfo[" + i + "]: len == " + c.getLen());
        }
    }

    private void acquireClient() throws IOException {
        if (this.xceiverClientFactory != null && this.xceiverClient == null) {
            Pipeline pipeline = this.pipelineRef.get();
            try {
                this.xceiverClient = this.xceiverClientFactory.acquireClientForReadData(pipeline);
            }
            catch (IOException ioe) {
                LOG.warn("Failed to acquire client for pipeline {}, block {}", (Object)pipeline, (Object)this.blockID);
                throw ioe;
            }
        }
    }

    protected synchronized void addStream(ContainerProtos.ChunkInfo chunkInfo) {
        this.chunkStreams.add(this.createChunkInputStream(chunkInfo));
    }

    protected ChunkInputStream createChunkInputStream(ContainerProtos.ChunkInfo chunkInfo) {
        return new ChunkInputStream(chunkInfo, this.blockID, this.xceiverClientFactory, this.pipelineRef::get, this.verifyChecksum, this.tokenRef::get);
    }

    @Override
    protected synchronized int readWithStrategy(ByteReaderStrategy strategy) throws IOException {
        Preconditions.checkArgument((strategy != null ? 1 : 0) != 0);
        if (!this.initialized) {
            this.initialize();
        }
        this.checkOpen();
        int totalReadLen = 0;
        int len = strategy.getTargetLength();
        while (len > 0) {
            int numBytesRead;
            if (this.chunkStreams.isEmpty() || this.chunkStreams.size() - 1 <= this.chunkIndex && this.chunkStreams.get(this.chunkIndex).getRemaining() == 0L) {
                return totalReadLen == 0 ? -1 : totalReadLen;
            }
            ChunkInputStream current = this.chunkStreams.get(this.chunkIndex);
            int numBytesToRead = Math.min(len, (int)current.getRemaining());
            try {
                numBytesRead = strategy.readFromBlock(current, numBytesToRead);
                this.retries = 0;
            }
            catch (SCMSecurityException ex) {
                throw ex;
            }
            catch (StorageContainerException e) {
                if (this.shouldRetryRead((IOException)((Object)e))) {
                    this.handleReadError((IOException)((Object)e));
                    continue;
                }
                throw e;
            }
            catch (IOException ex) {
                if (this.shouldRetryRead(ex)) {
                    if (this.isConnectivityIssue(ex)) {
                        this.handleReadError(ex);
                        continue;
                    }
                    current.releaseClient();
                    continue;
                }
                throw ex;
            }
            if (numBytesRead != numBytesToRead) {
                throw new IOException(String.format("Inconsistent read for chunkName=%s length=%d numBytesToRead= %d numBytesRead=%d", current.getChunkName(), current.getLength(), numBytesToRead, numBytesRead));
            }
            totalReadLen += numBytesRead;
            len -= numBytesRead;
            if (current.getRemaining() > 0L || this.chunkIndex + 1 >= this.chunkStreams.size()) continue;
            ++this.chunkIndex;
        }
        return totalReadLen;
    }

    @Override
    public synchronized void seek(long pos) throws IOException {
        if (!this.initialized) {
            this.blockPosition = pos;
            return;
        }
        this.checkOpen();
        if (pos < 0L || pos > this.length) {
            if (pos == 0L) {
                return;
            }
            throw new EOFException("EOF encountered at pos: " + pos + " for block: " + this.blockID);
        }
        if (this.chunkIndex >= this.chunkStreams.size()) {
            this.chunkIndex = Arrays.binarySearch(this.chunkOffsets, pos);
        } else if (pos < this.chunkOffsets[this.chunkIndex]) {
            this.chunkIndex = Arrays.binarySearch(this.chunkOffsets, 0, this.chunkIndex, pos);
        } else if (pos >= this.chunkOffsets[this.chunkIndex] + this.chunkStreams.get(this.chunkIndex).getLength()) {
            this.chunkIndex = Arrays.binarySearch(this.chunkOffsets, this.chunkIndex + 1, this.chunkStreams.size(), pos);
        }
        if (this.chunkIndex < 0) {
            this.chunkIndex = -this.chunkIndex - 2;
        }
        this.chunkStreams.get(this.chunkIndexOfPrevPosition).resetPosition();
        for (int index = this.chunkIndex + 1; index < this.chunkStreams.size(); ++index) {
            this.chunkStreams.get(index).seek(0L);
        }
        this.chunkStreams.get(this.chunkIndex).seek(pos - this.chunkOffsets[this.chunkIndex]);
        this.chunkIndexOfPrevPosition = this.chunkIndex;
    }

    @Override
    public synchronized long getPos() {
        if (this.length == 0L) {
            return 0L;
        }
        if (!this.initialized) {
            return this.blockPosition;
        }
        return this.chunkOffsets[this.chunkIndex] + this.chunkStreams.get(this.chunkIndex).getPos();
    }

    @Override
    public boolean seekToNewSource(long targetPos) throws IOException {
        return false;
    }

    @Override
    public synchronized void close() {
        this.releaseClient();
        this.xceiverClientFactory = null;
        List<ChunkInputStream> inputStreams = this.chunkStreams;
        if (inputStreams != null) {
            for (ChunkInputStream is : inputStreams) {
                is.close();
            }
        }
    }

    private void releaseClient() {
        if (this.xceiverClientFactory != null && this.xceiverClient != null) {
            this.xceiverClientFactory.releaseClientForReadData(this.xceiverClient, false);
            this.xceiverClient = null;
        }
    }

    protected synchronized void checkOpen() throws IOException {
        if (this.xceiverClientFactory == null) {
            throw new IOException("BlockInputStream has been closed.");
        }
    }

    @Override
    public BlockID getBlockID() {
        return this.blockID;
    }

    @Override
    public long getLength() {
        return this.length;
    }

    @VisibleForTesting
    synchronized int getChunkIndex() {
        return this.chunkIndex;
    }

    @VisibleForTesting
    synchronized long getBlockPosition() {
        return this.blockPosition;
    }

    public synchronized void unbuffer() {
        this.storePosition();
        this.releaseClient();
        List<ChunkInputStream> inputStreams = this.chunkStreams;
        if (inputStreams != null) {
            for (ChunkInputStream is : inputStreams) {
                is.unbuffer();
            }
        }
    }

    private synchronized void storePosition() {
        this.blockPosition = this.getPos();
    }

    private boolean shouldRetryRead(IOException cause) throws IOException {
        RetryPolicy.RetryAction retryAction;
        try {
            retryAction = this.retryPolicy.shouldRetry((Exception)cause, ++this.retries, 0, true);
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IOException(e);
        }
        if (retryAction.action == RetryPolicy.RetryAction.RetryDecision.RETRY) {
            if (retryAction.delayMillis > 0L) {
                try {
                    LOG.debug("Retry read after {}ms", (Object)retryAction.delayMillis);
                    Thread.sleep(retryAction.delayMillis);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    String msg = "Interrupted: action=" + retryAction.action + ", retry policy=" + this.retryPolicy;
                    throw new IOException(msg, e);
                }
            }
            return true;
        }
        return false;
    }

    private void handleReadError(IOException cause) throws IOException {
        this.releaseClient();
        List<ChunkInputStream> inputStreams = this.chunkStreams;
        if (inputStreams != null) {
            for (ChunkInputStream is : inputStreams) {
                is.releaseClient();
            }
        }
        this.refreshBlockInfo(cause);
    }

    @VisibleForTesting
    public synchronized List<ChunkInputStream> getChunkStreams() {
        return this.chunkStreams;
    }
}

