/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.storageengine.buffer;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.function.LongConsumer;
import org.apache.iotdb.commons.exception.IoTDBIORuntimeException;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.storageengine.dataregion.read.control.FileReaderManager;
import org.apache.iotdb.db.storageengine.dataregion.tsfile.TsFileID;
import org.apache.tsfile.read.TsFileSequenceReader;
import org.apache.tsfile.utils.BloomFilter;
import org.apache.tsfile.utils.RamUsageEstimator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BloomFilterCache {
    private static final Logger LOGGER = LoggerFactory.getLogger(BloomFilterCache.class);
    private static final Logger DEBUG_LOGGER = LoggerFactory.getLogger((String)"QUERY_DEBUG");
    private static final IoTDBConfig CONFIG = IoTDBDescriptor.getInstance().getConfig();
    private static final long MEMORY_THRESHOLD_IN_BLOOM_FILTER_CACHE = CONFIG.getAllocateMemoryForBloomFilterCache();
    private static final boolean CACHE_ENABLE = CONFIG.isMetaDataCacheEnable();
    private final AtomicLong entryAverageSize = new AtomicLong(0L);
    private final Cache<BloomFilterCacheKey, BloomFilter> lruCache;

    private BloomFilterCache() {
        if (CACHE_ENABLE) {
            LOGGER.info("BloomFilterCache size = {}", (Object)MEMORY_THRESHOLD_IN_BLOOM_FILTER_CACHE);
        }
        this.lruCache = Caffeine.newBuilder().maximumWeight(MEMORY_THRESHOLD_IN_BLOOM_FILTER_CACHE).weigher((key, bloomFilter) -> (int)(key.getRetainedSizeInBytes() + bloomFilter.getRetainedSizeInBytes())).recordStats().build();
    }

    public static BloomFilterCache getInstance() {
        return BloomFilterCacheHolder.INSTANCE;
    }

    public BloomFilter get(BloomFilterCacheKey key) throws IOException {
        LongConsumer emptyConsumer = l -> {};
        return this.get(key, false, emptyConsumer, emptyConsumer, emptyConsumer);
    }

    public BloomFilter get(BloomFilterCacheKey key, boolean debug, LongConsumer ioSizeRecorder, LongConsumer cacheHitAdder, LongConsumer cacheMissAdder) throws IOException {
        BloomFilterLoader loader = new BloomFilterLoader(ioSizeRecorder);
        try {
            if (!CACHE_ENABLE) {
                BloomFilter bloomFilter = loader.apply(key);
                return bloomFilter;
            }
            BloomFilter bloomFilter = (BloomFilter)this.lruCache.get((Object)key, (Function)loader);
            if (debug) {
                DEBUG_LOGGER.info("get bloomFilter from cache where filePath is: {}", (Object)key.filePath);
            }
            BloomFilter bloomFilter2 = bloomFilter;
            return bloomFilter2;
        }
        catch (IoTDBIORuntimeException e) {
            throw e.getCause();
        }
        finally {
            if (loader.isCacheMiss()) {
                cacheMissAdder.accept(1L);
            } else {
                cacheHitAdder.accept(1L);
            }
        }
    }

    public double calculateBloomFilterHitRatio() {
        return this.lruCache.stats().hitRate();
    }

    public long getEvictionCount() {
        return this.lruCache.stats().evictionCount();
    }

    public long getMaxMemory() {
        return MEMORY_THRESHOLD_IN_BLOOM_FILTER_CACHE;
    }

    public double getAverageLoadPenalty() {
        return this.lruCache.stats().averageLoadPenalty();
    }

    public long getAverageSize() {
        return this.entryAverageSize.get();
    }

    public void clear() {
        this.lruCache.invalidateAll();
        this.lruCache.cleanUp();
    }

    public void remove(BloomFilterCacheKey key) {
        this.lruCache.invalidate((Object)key);
    }

    public BloomFilter getIfPresent(BloomFilterCacheKey key) {
        return (BloomFilter)this.lruCache.getIfPresent((Object)key);
    }

    private static class BloomFilterCacheHolder {
        private static final BloomFilterCache INSTANCE = new BloomFilterCache();

        private BloomFilterCacheHolder() {
        }
    }

    public static class BloomFilterCacheKey {
        private static final long INSTANCE_SIZE = RamUsageEstimator.shallowSizeOfInstance(BloomFilterCacheKey.class);
        private final String filePath;
        private final TsFileID tsFileID;

        public BloomFilterCacheKey(String filePath, TsFileID tsFileID) {
            this.filePath = filePath;
            this.tsFileID = tsFileID;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            BloomFilterCacheKey that = (BloomFilterCacheKey)o;
            return Objects.equals(this.tsFileID, that.tsFileID);
        }

        public int hashCode() {
            return Objects.hash(this.tsFileID);
        }

        public long getRetainedSizeInBytes() {
            return INSTANCE_SIZE;
        }
    }

    private static class BloomFilterLoader
    implements Function<BloomFilterCacheKey, BloomFilter> {
        private boolean cacheMiss = false;
        private final LongConsumer ioSizeRecorder;

        private BloomFilterLoader(LongConsumer ioSizeRecorder) {
            this.ioSizeRecorder = ioSizeRecorder;
        }

        @Override
        public BloomFilter apply(BloomFilterCacheKey bloomFilterCacheKey) {
            try {
                this.cacheMiss = true;
                TsFileSequenceReader reader = FileReaderManager.getInstance().get(bloomFilterCacheKey.filePath, bloomFilterCacheKey.tsFileID, true, this.ioSizeRecorder);
                return reader.readBloomFilter(this.ioSizeRecorder);
            }
            catch (IOException e) {
                throw new IoTDBIORuntimeException(e);
            }
        }

        public boolean isCacheMiss() {
            return this.cacheMiss;
        }
    }
}

