/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gravitino.storage.kv;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.SerializationUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.gravitino.Config;
import org.apache.gravitino.Configs;
import org.apache.gravitino.Entity;
import org.apache.gravitino.NameIdentifier;
import org.apache.gravitino.storage.EntityKeyEncoder;
import org.apache.gravitino.storage.kv.KvBackend;
import org.apache.gravitino.storage.kv.KvNameMappingService;
import org.apache.gravitino.storage.kv.KvRange;
import org.apache.gravitino.storage.kv.TransactionalKvBackendImpl;
import org.apache.gravitino.utils.Bytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KvGarbageCollector
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(KvGarbageCollector.class);
    private final KvBackend kvBackend;
    private final Config config;
    private final EntityKeyEncoder<byte[]> entityKeyEncoder;
    private static final byte[] LAST_COLLECT_COMMIT_ID_KEY = Bytes.concat({29, 0, 3}, "last_collect_commit_id".getBytes(StandardCharsets.UTF_8));
    byte[] commitIdHasBeenCollected;
    private long frequencyInMinutes;
    private static final String TIME_STAMP_FORMAT = "yyyy-MM-dd HH:mm:ss.SSS";
    @VisibleForTesting
    final ScheduledExecutorService garbageCollectorPool = new ScheduledThreadPoolExecutor(2, r -> {
        Thread t = new Thread(r, "KvEntityStore-Garbage-Collector");
        t.setDaemon(true);
        return t;
    }, new ThreadPoolExecutor.AbortPolicy());

    public KvGarbageCollector(KvBackend kvBackend, Config config, EntityKeyEncoder<byte[]> entityKeyEncoder) {
        this.kvBackend = kvBackend;
        this.config = config;
        this.entityKeyEncoder = entityKeyEncoder;
    }

    public void start() {
        long dateTimelineMinute = this.config.get(Configs.STORE_DELETE_AFTER_TIME) / 1000L / 60L;
        this.frequencyInMinutes = Math.max(dateTimelineMinute / 10L, 10L);
        this.garbageCollectorPool.scheduleAtFixedRate(this::collectAndClean, 5L, this.frequencyInMinutes, TimeUnit.MINUTES);
    }

    @VisibleForTesting
    void collectAndClean() {
        LOG.info("Start to collect garbage...");
        try {
            LOG.info("Start to collect and delete uncommitted data...");
            this.collectAndRemoveUncommittedData();
            LOG.info("Start to collect and delete old version data...");
            this.collectAndRemoveOldVersionData();
        }
        catch (Exception e) {
            LOG.error("Failed to collect garbage", (Throwable)e);
        }
    }

    private void collectAndRemoveUncommittedData() throws IOException {
        List<Pair<byte[], byte[]>> kvs = this.kvBackend.scan(new KvRange.KvRangeBuilder().start(new byte[]{32}).end(new byte[]{127}).startInclusive(true).endInclusive(false).predicate((k, v) -> {
            byte[] transactionId = TransactionalKvBackendImpl.getBinaryTransactionId(k);
            long writeTime = TransactionalKvBackendImpl.getTransactionId(transactionId) >> 18;
            if (writeTime < System.currentTimeMillis() - this.frequencyInMinutes * 60L * 1000L * 2L) {
                return false;
            }
            return this.kvBackend.get(TransactionalKvBackendImpl.generateCommitKey(transactionId)) == null;
        }).limit(10000).build());
        LOG.info("Start to remove {} uncommitted data", (Object)kvs.size());
        for (Pair<byte[], byte[]> pair : kvs) {
            LogHelper logHelper = this.decodeKey((byte[])pair.getKey());
            LOG.info("Physically delete key that has marked uncommitted: name identity: '{}', entity type: '{}', createTime: '{}({})', key: '{}'", new Object[]{logHelper.identifier, logHelper.type, logHelper.createTimeAsString, logHelper.createTimeInMs, pair.getKey()});
            this.kvBackend.delete((byte[])pair.getKey());
        }
    }

    private void collectAndRemoveOldVersionData() throws IOException {
        long deleteTimeline = System.currentTimeMillis() - this.config.get(Configs.STORE_DELETE_AFTER_TIME);
        long transactionIdToDelete = deleteTimeline << 18;
        LOG.info("Start to remove data which is older than {}", (Object)transactionIdToDelete);
        byte[] startKey = TransactionalKvBackendImpl.generateCommitKey(transactionIdToDelete);
        this.commitIdHasBeenCollected = this.kvBackend.get(LAST_COLLECT_COMMIT_ID_KEY);
        if (this.commitIdHasBeenCollected == null) {
            this.commitIdHasBeenCollected = TransactionalKvBackendImpl.endOfTransactionId();
        }
        long lastGCId = TransactionalKvBackendImpl.getTransactionId(TransactionalKvBackendImpl.getBinaryTransactionId(this.commitIdHasBeenCollected));
        LOG.info("Start to collect data which is modified between '{}({})' (exclusive) and '{}({})' (inclusive)", new Object[]{lastGCId, lastGCId == 1L ? Long.valueOf(lastGCId) : DateFormatUtils.format((long)(lastGCId >> 18), (String)TIME_STAMP_FORMAT), transactionIdToDelete, DateFormatUtils.format((long)deleteTimeline, (String)TIME_STAMP_FORMAT)});
        List<Pair<byte[], byte[]>> kvs = this.kvBackend.scan(new KvRange.KvRangeBuilder().start(startKey).end(this.commitIdHasBeenCollected).startInclusive(true).endInclusive(false).build());
        for (Pair<byte[], byte[]> kv : kvs) {
            List keysInTheTransaction = (List)SerializationUtils.deserialize((byte[])((byte[])kv.getValue()));
            byte[] transactionId = TransactionalKvBackendImpl.getBinaryTransactionId((byte[])kv.getKey());
            int keysDeletedCount = 0;
            for (byte[] key : keysInTheTransaction) {
                byte[] rawKey = TransactionalKvBackendImpl.generateKey(key, transactionId);
                byte[] rawValue = this.kvBackend.get(rawKey);
                if (null == rawValue) {
                    ++keysDeletedCount;
                    continue;
                }
                if (null == TransactionalKvBackendImpl.getRealValue(rawValue)) {
                    this.removeAllVersionsOfKey(rawKey, key, false);
                    LogHelper logHelper = this.decodeKey(key, transactionId);
                    this.kvBackend.delete(rawKey);
                    LOG.info("Physically delete key that has marked deleted: name identifier: '{}', entity type: '{}', createTime: '{}({})', key: '{}'", new Object[]{logHelper.identifier, logHelper.type, logHelper.createTimeAsString, logHelper.createTimeInMs, Bytes.wrap(key)});
                    ++keysDeletedCount;
                    continue;
                }
                List<Pair<byte[], byte[]>> newVersionOfKey = this.kvBackend.scan(new KvRange.KvRangeBuilder().start(key).end(TransactionalKvBackendImpl.generateKey(key, transactionId)).startInclusive(false).endInclusive(false).limit(1).build());
                if (newVersionOfKey.isEmpty()) continue;
                this.removeAllVersionsOfKey(rawKey, key, false);
                LogHelper logHelper = this.decodeKey(key, transactionId);
                byte[] newVersionKey = (byte[])newVersionOfKey.get(0).getKey();
                LogHelper newVersionLogHelper = this.decodeKey(newVersionKey);
                this.kvBackend.delete(rawKey);
                LOG.info("Physically delete key that has newer version: name identifier: '{}', entity type: '{}', createTime: '{}({})', newVersion createTime: '{}({})', key: '{}', newVersion key: '{}'", new Object[]{logHelper.identifier, logHelper.type, logHelper.createTimeAsString, logHelper.createTimeInMs, newVersionLogHelper.createTimeAsString, newVersionLogHelper.createTimeInMs, Bytes.wrap(rawKey), Bytes.wrap(newVersionKey)});
                ++keysDeletedCount;
            }
            if (keysDeletedCount != keysInTheTransaction.size()) continue;
            this.kvBackend.delete((byte[])kv.getKey());
            long timestamp = TransactionalKvBackendImpl.getTransactionId(transactionId) >> 18;
            LOG.info("Physically delete commit mark: {}, createTime: '{}({})', key: '{}'", new Object[]{Bytes.wrap((byte[])kv.getKey()), DateFormatUtils.format((long)timestamp, (String)TIME_STAMP_FORMAT), timestamp, Bytes.wrap((byte[])kv.getKey())});
        }
        this.commitIdHasBeenCollected = kvs.isEmpty() ? startKey : (byte[])kvs.get(0).getKey();
        this.kvBackend.put(LAST_COLLECT_COMMIT_ID_KEY, this.commitIdHasBeenCollected, true);
    }

    private void removeAllVersionsOfKey(byte[] rawKey, byte[] key, boolean includeStart) throws IOException {
        List<Pair<byte[], byte[]>> kvs = this.kvBackend.scan(new KvRange.KvRangeBuilder().start(rawKey).end(TransactionalKvBackendImpl.generateKey(key, 1L)).startInclusive(includeStart).endInclusive(false).build());
        for (Pair<byte[], byte[]> kv : kvs) {
            this.kvBackend.delete((byte[])kv.getKey());
            LogHelper logHelper = this.decodeKey((byte[])kv.getKey());
            LOG.info("Physically delete key that has marked deleted: name identifier: '{}', entity type: '{}', createTime: '{}({})', key: '{}'", new Object[]{logHelper.identifier, logHelper.type, logHelper.createTimeAsString, logHelper.createTimeInMs, Bytes.wrap(key)});
            byte[] transactionId = TransactionalKvBackendImpl.getBinaryTransactionId((byte[])kv.getKey());
            byte[] transactionKey = TransactionalKvBackendImpl.generateCommitKey(transactionId);
            byte[] transactionValue = this.kvBackend.get(transactionKey);
            List keysInTheTransaction = (List)SerializationUtils.deserialize((byte[])transactionValue);
            boolean allDropped = true;
            for (byte[] keyInTheTransaction : keysInTheTransaction) {
                if (this.kvBackend.get(TransactionalKvBackendImpl.generateKey(keyInTheTransaction, transactionId)) == null) continue;
                allDropped = false;
                break;
            }
            if (!allDropped) continue;
            this.kvBackend.delete(transactionKey);
            long timestamp = TransactionalKvBackendImpl.getTransactionId(transactionId) >> 18;
            LOG.info("Physically delete commit mark: {}, createTime: '{}({})', key: '{}'", new Object[]{Bytes.wrap((byte[])kv.getKey()), DateFormatUtils.format((long)timestamp, (String)TIME_STAMP_FORMAT), timestamp, Bytes.wrap((byte[])kv.getKey())});
        }
    }

    @VisibleForTesting
    LogHelper decodeKey(byte[] key, byte[] timestampArray) {
        Pair<NameIdentifier, Entity.EntityType> entityTypePair;
        if (this.entityKeyEncoder == null) {
            return LogHelper.NONE;
        }
        if (Arrays.equals(KvNameMappingService.GENERAL_NAME_MAPPING_PREFIX, ArrayUtils.subarray((byte[])key, (int)0, (int)3))) {
            return LogHelper.NONE;
        }
        try {
            entityTypePair = this.entityKeyEncoder.decode(key);
        }
        catch (Exception e) {
            LOG.warn("Unable to decode key: {}", (Object)Bytes.wrap(key), (Object)e);
            return LogHelper.NONE;
        }
        long timestamp = TransactionalKvBackendImpl.getTransactionId(timestampArray) >> 18;
        String ts = DateFormatUtils.format((long)timestamp, (String)TIME_STAMP_FORMAT);
        return new LogHelper((NameIdentifier)entityTypePair.getKey(), (Entity.EntityType)((Object)entityTypePair.getValue()), timestamp, ts);
    }

    @VisibleForTesting
    LogHelper decodeKey(byte[] rawKey) {
        byte[] key = TransactionalKvBackendImpl.getRealKey(rawKey);
        byte[] timestampArray = TransactionalKvBackendImpl.getBinaryTransactionId(rawKey);
        return this.decodeKey(key, timestampArray);
    }

    @Override
    public void close() throws IOException {
        this.garbageCollectorPool.shutdownNow();
        try {
            this.garbageCollectorPool.awaitTermination(5L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.error("Failed to close garbage collector", (Throwable)e);
        }
    }

    static class LogHelper {
        @VisibleForTesting
        final NameIdentifier identifier;
        @VisibleForTesting
        final Entity.EntityType type;
        @VisibleForTesting
        final long createTimeInMs;
        @VisibleForTesting
        final String createTimeAsString;
        public static final LogHelper NONE = new LogHelper(null, null, 0L, null);

        public LogHelper(NameIdentifier identifier, Entity.EntityType type, long createTimeInMs, String createTimeAsString) {
            this.identifier = identifier;
            this.type = type;
            this.createTimeInMs = createTimeInMs;
            this.createTimeAsString = createTimeAsString;
        }
    }
}

