/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.server.kerberos.kdc.authentication;

import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.Date;
import java.util.List;
import java.util.Set;
import javax.security.auth.kerberos.KerberosKey;
import javax.security.auth.kerberos.KerberosPrincipal;
import org.apache.directory.api.asn1.Asn1Object;
import org.apache.directory.api.asn1.EncoderException;
import org.apache.directory.api.ldap.model.constants.Loggers;
import org.apache.directory.server.i18n.I18n;
import org.apache.directory.server.kerberos.KerberosConfig;
import org.apache.directory.server.kerberos.kdc.KdcContext;
import org.apache.directory.server.kerberos.kdc.authentication.AuthenticationContext;
import org.apache.directory.server.kerberos.sam.SamException;
import org.apache.directory.server.kerberos.sam.SamSubsystem;
import org.apache.directory.server.kerberos.shared.crypto.encryption.CipherTextHandler;
import org.apache.directory.server.kerberos.shared.crypto.encryption.KeyUsage;
import org.apache.directory.server.kerberos.shared.crypto.encryption.RandomKeyFactory;
import org.apache.directory.server.kerberos.shared.store.PrincipalStore;
import org.apache.directory.server.kerberos.shared.store.PrincipalStoreEntry;
import org.apache.directory.shared.kerberos.KerberosTime;
import org.apache.directory.shared.kerberos.KerberosUtils;
import org.apache.directory.shared.kerberos.codec.KerberosDecoder;
import org.apache.directory.shared.kerberos.codec.options.KdcOptions;
import org.apache.directory.shared.kerberos.codec.types.EncryptionType;
import org.apache.directory.shared.kerberos.codec.types.LastReqType;
import org.apache.directory.shared.kerberos.codec.types.PaDataType;
import org.apache.directory.shared.kerberos.components.ETypeInfo;
import org.apache.directory.shared.kerberos.components.ETypeInfo2;
import org.apache.directory.shared.kerberos.components.ETypeInfo2Entry;
import org.apache.directory.shared.kerberos.components.ETypeInfoEntry;
import org.apache.directory.shared.kerberos.components.EncKdcRepPart;
import org.apache.directory.shared.kerberos.components.EncTicketPart;
import org.apache.directory.shared.kerberos.components.EncryptedData;
import org.apache.directory.shared.kerberos.components.EncryptionKey;
import org.apache.directory.shared.kerberos.components.KdcReq;
import org.apache.directory.shared.kerberos.components.KdcReqBody;
import org.apache.directory.shared.kerberos.components.LastReq;
import org.apache.directory.shared.kerberos.components.LastReqEntry;
import org.apache.directory.shared.kerberos.components.MethodData;
import org.apache.directory.shared.kerberos.components.PaData;
import org.apache.directory.shared.kerberos.components.PaEncTsEnc;
import org.apache.directory.shared.kerberos.components.PrincipalName;
import org.apache.directory.shared.kerberos.components.TransitedEncoding;
import org.apache.directory.shared.kerberos.exceptions.ErrorType;
import org.apache.directory.shared.kerberos.exceptions.KerberosException;
import org.apache.directory.shared.kerberos.flags.KerberosFlag;
import org.apache.directory.shared.kerberos.flags.TicketFlag;
import org.apache.directory.shared.kerberos.flags.TicketFlags;
import org.apache.directory.shared.kerberos.messages.AsRep;
import org.apache.directory.shared.kerberos.messages.EncAsRepPart;
import org.apache.directory.shared.kerberos.messages.KerberosMessage;
import org.apache.directory.shared.kerberos.messages.Ticket;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class AuthenticationService {
    private static final Logger LOG_KRB = LoggerFactory.getLogger((String)Loggers.KERBEROS_LOG.getName());
    private static final CipherTextHandler CIPHER_TEXT_HANDLER = new CipherTextHandler();
    private static final String SERVICE_NAME = "Authentication Service (AS)";

    private AuthenticationService() {
    }

    public static void execute(AuthenticationContext authContext) throws Exception {
        if (LOG_KRB.isDebugEnabled()) {
            AuthenticationService.monitorRequest(authContext);
        }
        authContext.setCipherTextHandler(CIPHER_TEXT_HANDLER);
        int kerberosVersion = authContext.getRequest().getProtocolVersionNumber();
        if (kerberosVersion != 5) {
            LOG_KRB.error("Kerberos V{} is not supported", (Object)kerberosVersion);
            throw new KerberosException(ErrorType.KDC_ERR_BAD_PVNO);
        }
        AuthenticationService.selectEncryptionType(authContext);
        AuthenticationService.getClientEntry(authContext);
        AuthenticationService.verifyPolicy(authContext);
        AuthenticationService.verifySam(authContext);
        AuthenticationService.verifyEncryptedTimestamp(authContext);
        AuthenticationService.getServerEntry(authContext);
        AuthenticationService.generateTicket(authContext);
        AuthenticationService.buildReply(authContext);
    }

    private static void selectEncryptionType(AuthenticationContext authContext) throws KerberosException {
        LOG_KRB.debug("--> Selecting the EncryptionType");
        AuthenticationContext kdcContext = authContext;
        KerberosConfig config = kdcContext.getConfig();
        Set requestedTypes = kdcContext.getRequest().getKdcReqBody().getEType();
        LOG_KRB.debug("Encryption types requested by client {}.", (Object)requestedTypes);
        EncryptionType bestType = KerberosUtils.getBestEncryptionType((Set)requestedTypes, config.getEncryptionTypes());
        LOG_KRB.debug("Session will use encryption type {}.", (Object)bestType);
        if (bestType == null) {
            LOG_KRB.error("No encryptionType selected !");
            throw new KerberosException(ErrorType.KDC_ERR_ETYPE_NOSUPP);
        }
        kdcContext.setEncryptionType(bestType);
    }

    private static void getClientEntry(AuthenticationContext authContext) throws KerberosException {
        LOG_KRB.debug("--> Getting the client Entry");
        KdcReqBody kdcReqBody = authContext.getRequest().getKdcReqBody();
        KerberosPrincipal principal = KerberosUtils.getKerberosPrincipal((PrincipalName)kdcReqBody.getCName(), (String)kdcReqBody.getRealm());
        PrincipalStore store = authContext.getStore();
        try {
            PrincipalStoreEntry storeEntry = KerberosUtils.getEntry((KerberosPrincipal)principal, (PrincipalStore)store, (ErrorType)ErrorType.KDC_ERR_C_PRINCIPAL_UNKNOWN);
            authContext.setClientEntry(storeEntry);
            LOG_KRB.debug("Found entry {} for principal {}", (Object)storeEntry.getDistinguishedName(), (Object)principal);
        }
        catch (KerberosException ke) {
            LOG_KRB.error("Error while searching for client {} : {}", (Object)principal, (Object)ke.getMessage());
            throw ke;
        }
    }

    private static void verifyPolicy(AuthenticationContext authContext) throws KerberosException {
        LOG_KRB.debug("--> Verifying the policy");
        PrincipalStoreEntry entry = authContext.getClientEntry();
        if (entry.isDisabled()) {
            LOG_KRB.error("The entry {} is disabled", (Object)entry.getDistinguishedName());
            throw new KerberosException(ErrorType.KDC_ERR_CLIENT_REVOKED);
        }
        if (entry.isLockedOut()) {
            LOG_KRB.error("The entry {} is locked out", (Object)entry.getDistinguishedName());
            throw new KerberosException(ErrorType.KDC_ERR_CLIENT_REVOKED);
        }
        if (entry.getExpiration().getTime() < new Date().getTime()) {
            LOG_KRB.error("The entry {} has been revoked", (Object)entry.getDistinguishedName());
            throw new KerberosException(ErrorType.KDC_ERR_CLIENT_REVOKED);
        }
    }

    private static void verifySam(AuthenticationContext authContext) throws KerberosException {
        LOG_KRB.debug("--> Verifying using SAM subsystem.");
        KdcReq request = authContext.getRequest();
        KerberosConfig config = authContext.getConfig();
        PrincipalStoreEntry clientEntry = authContext.getClientEntry();
        String clientName = clientEntry.getPrincipal().getName();
        EncryptionKey clientKey = null;
        if (clientEntry.getSamType() != null) {
            List preAuthData;
            if (LOG_KRB.isDebugEnabled()) {
                LOG_KRB.debug("Entry for client principal {} has a valid SAM type.  Invoking SAM subsystem for pre-authentication.", (Object)clientName);
            }
            if ((preAuthData = request.getPaData()) == null || preAuthData.isEmpty()) {
                LOG_KRB.debug("No PreAuth Data");
                throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_REQUIRED, AuthenticationService.preparePreAuthenticationError(authContext.getEncryptionType(), config.getEncryptionTypes()));
            }
            try {
                for (PaData paData : preAuthData) {
                    if (!paData.getPaDataType().equals((Object)PaDataType.PA_ENC_TIMESTAMP)) continue;
                    KerberosKey samKey = SamSubsystem.getInstance().verify(clientEntry, paData.getPaDataValue());
                    clientKey = new EncryptionKey(EncryptionType.getTypeByValue((int)samKey.getKeyType()), samKey.getEncoded());
                }
            }
            catch (SamException se) {
                LOG_KRB.error("Error : {}", (Object)se.getMessage());
                throw new KerberosException(ErrorType.KRB_ERR_GENERIC, (Throwable)se);
            }
            authContext.setClientKey(clientKey);
            authContext.setPreAuthenticated(true);
            if (LOG_KRB.isDebugEnabled()) {
                LOG_KRB.debug("Pre-authentication using SAM subsystem successful for {}.", (Object)clientName);
            }
        }
    }

    private static void verifyEncryptedTimestamp(AuthenticationContext authContext) throws KerberosException {
        LOG_KRB.debug("--> Verifying using encrypted timestamp.");
        KerberosConfig config = authContext.getConfig();
        KdcReq request = authContext.getRequest();
        CipherTextHandler cipherTextHandler = authContext.getCipherTextHandler();
        PrincipalStoreEntry clientEntry = authContext.getClientEntry();
        String clientName = clientEntry.getPrincipal().getName();
        EncryptionKey clientKey = null;
        if (clientEntry.getSamType() == null) {
            LOG_KRB.debug("Entry for client principal {} has no SAM type.  Proceeding with standard pre-authentication.", (Object)clientName);
            EncryptionType encryptionType = authContext.getEncryptionType();
            clientKey = (EncryptionKey)clientEntry.getKeyMap().get(encryptionType);
            if (clientKey == null) {
                LOG_KRB.error("No key for client {}", (Object)clientEntry.getDistinguishedName());
                throw new KerberosException(ErrorType.KDC_ERR_NULL_KEY);
            }
            if (config.isPaEncTimestampRequired()) {
                List preAuthData = request.getPaData();
                if (preAuthData == null) {
                    LOG_KRB.debug("PRE_AUTH required...");
                    throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_REQUIRED, AuthenticationService.preparePreAuthenticationError(authContext.getEncryptionType(), config.getEncryptionTypes()));
                }
                PaEncTsEnc timestamp = null;
                for (PaData paData : preAuthData) {
                    if (!paData.getPaDataType().equals((Object)PaDataType.PA_ENC_TIMESTAMP)) continue;
                    EncryptedData dataValue = KerberosDecoder.decodeEncryptedData((byte[])paData.getPaDataValue());
                    byte[] decryptedData = cipherTextHandler.decrypt(clientKey, dataValue, KeyUsage.AS_REQ_PA_ENC_TIMESTAMP_WITH_CKEY);
                    timestamp = KerberosDecoder.decodePaEncTsEnc((byte[])decryptedData);
                }
                if (timestamp == null) {
                    LOG_KRB.error("No timestamp found");
                    throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_REQUIRED, AuthenticationService.preparePreAuthenticationError(authContext.getEncryptionType(), config.getEncryptionTypes()));
                }
                if (!timestamp.getPaTimestamp().isInClockSkew(config.getAllowableClockSkew())) {
                    LOG_KRB.error("Timestamp not in delay");
                    throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_FAILED);
                }
            }
        }
        authContext.setClientKey(clientKey);
        authContext.setPreAuthenticated(true);
        if (LOG_KRB.isDebugEnabled()) {
            LOG_KRB.debug("Pre-authentication by encrypted timestamp successful for {}.", (Object)clientName);
        }
    }

    private static void getServerEntry(AuthenticationContext authContext) throws KerberosException {
        PrincipalName principal = authContext.getRequest().getKdcReqBody().getSName();
        PrincipalStore store = authContext.getStore();
        LOG_KRB.debug("--> Getting the server entry for {}", (Object)principal);
        KerberosPrincipal principalWithRealm = new KerberosPrincipal(principal.getNameString() + "@" + authContext.getRequest().getKdcReqBody().getRealm());
        authContext.setServerEntry(KerberosUtils.getEntry((KerberosPrincipal)principalWithRealm, (PrincipalStore)store, (ErrorType)ErrorType.KDC_ERR_S_PRINCIPAL_UNKNOWN));
    }

    private static void generateTicket(AuthenticationContext authContext) throws KerberosException {
        KdcOptions kdcOptions;
        KdcReq request = authContext.getRequest();
        CipherTextHandler cipherTextHandler = authContext.getCipherTextHandler();
        PrincipalName serverPrincipal = request.getKdcReqBody().getSName();
        LOG_KRB.debug("--> Generating ticket for {}", (Object)serverPrincipal);
        EncryptionType encryptionType = authContext.getEncryptionType();
        EncryptionKey serverKey = (EncryptionKey)authContext.getServerEntry().getKeyMap().get(encryptionType);
        PrincipalName ticketPrincipal = request.getKdcReqBody().getSName();
        EncTicketPart encTicketPart = new EncTicketPart();
        KerberosConfig config = authContext.getConfig();
        TicketFlags ticketFlags = new TicketFlags();
        encTicketPart.setFlags(ticketFlags);
        ticketFlags.setFlag((KerberosFlag)TicketFlag.INITIAL);
        if (authContext.isPreAuthenticated()) {
            ticketFlags.setFlag((KerberosFlag)TicketFlag.PRE_AUTHENT);
        }
        if (request.getKdcReqBody().getKdcOptions().get(1)) {
            if (!config.isForwardableAllowed()) {
                LOG_KRB.error("Ticket cannot be generated, because Forwadable is not allowed");
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            ticketFlags.setFlag((KerberosFlag)TicketFlag.FORWARDABLE);
        }
        if (request.getKdcReqBody().getKdcOptions().get(3)) {
            if (!config.isProxiableAllowed()) {
                LOG_KRB.error("Ticket cannot be generated, because proxyiable is not allowed");
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            ticketFlags.setFlag((KerberosFlag)TicketFlag.PROXIABLE);
        }
        if (request.getKdcReqBody().getKdcOptions().get(5)) {
            if (!config.isPostdatedAllowed()) {
                LOG_KRB.error("Ticket cannot be generated, because Posdate is not allowed");
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            ticketFlags.setFlag((KerberosFlag)TicketFlag.MAY_POSTDATE);
        }
        if ((kdcOptions = request.getKdcReqBody().getKdcOptions()).get(30) || kdcOptions.get(31) || kdcOptions.get(4) || kdcOptions.get(2) || kdcOptions.get(28)) {
            String msg = "";
            if (kdcOptions.get(30)) {
                msg = "Ticket cannot be generated, as it's a renew";
            }
            if (kdcOptions.get(31)) {
                msg = "Ticket cannot be generated, as it's a validate";
            }
            if (kdcOptions.get(4)) {
                msg = "Ticket cannot be generated, as it's a proxy";
            }
            if (kdcOptions.get(2)) {
                msg = "Ticket cannot be generated, as it's forwarded";
            }
            if (kdcOptions.get(28)) {
                msg = "Ticket cannot be generated, as it's a user-to-user ";
            }
            if (LOG_KRB.isDebugEnabled()) {
                LOG_KRB.debug(msg);
            }
            throw new KerberosException(ErrorType.KDC_ERR_BADOPTION, msg);
        }
        EncryptionKey sessionKey = RandomKeyFactory.getRandomKey((EncryptionType)authContext.getEncryptionType());
        encTicketPart.setKey(sessionKey);
        encTicketPart.setCName(request.getKdcReqBody().getCName());
        encTicketPart.setCRealm(request.getKdcReqBody().getRealm());
        encTicketPart.setTransited(new TransitedEncoding());
        String serverRealm = request.getKdcReqBody().getRealm();
        KerberosTime now = new KerberosTime();
        encTicketPart.setAuthTime(now);
        KerberosTime startTime = request.getKdcReqBody().getFrom();
        if (startTime == null || startTime.lessThan(now) || startTime.isInClockSkew(config.getAllowableClockSkew()) && !request.getKdcReqBody().getKdcOptions().get(6)) {
            startTime = now;
        }
        if (startTime != null && startTime.greaterThan(now) && !startTime.isInClockSkew(config.getAllowableClockSkew()) && !request.getKdcReqBody().getKdcOptions().get(6)) {
            String msg = "Ticket cannot be generated, as it's in the future and the POSTDATED option is not set in the request";
            LOG_KRB.error(msg);
            throw new KerberosException(ErrorType.KDC_ERR_CANNOT_POSTDATE, msg);
        }
        if (request.getKdcReqBody().getKdcOptions().get(6)) {
            if (!config.isPostdatedAllowed()) {
                String msg = "Ticket cannot be generated, cause issuing POSTDATED tickets is not allowed";
                LOG_KRB.error(msg);
                throw new KerberosException(ErrorType.KDC_ERR_POLICY, msg);
            }
            ticketFlags.setFlag((KerberosFlag)TicketFlag.POSTDATED);
            ticketFlags.setFlag((KerberosFlag)TicketFlag.INVALID);
        }
        encTicketPart.setStartTime(startTime);
        long till = 0L;
        till = request.getKdcReqBody().getTill().getTime() == 0L ? Long.MAX_VALUE : request.getKdcReqBody().getTill().getTime();
        long endTime = Math.min(till, startTime.getTime() + config.getMaximumTicketLifetime());
        KerberosTime kerberosEndTime = new KerberosTime(endTime);
        encTicketPart.setEndTime(kerberosEndTime);
        if (kerberosEndTime.lessThan(startTime)) {
            String msg = "Ticket cannot be generated, as the endTime is below the startTime";
            LOG_KRB.error(msg);
            throw new KerberosException(ErrorType.KDC_ERR_NEVER_VALID, msg);
        }
        long ticketLifeTime = Math.abs(startTime.getTime() - kerberosEndTime.getTime());
        if (ticketLifeTime < config.getMinimumTicketLifetime()) {
            String msg = "Ticket cannot be generated, as the Lifetime is too small";
            LOG_KRB.error(msg);
            throw new KerberosException(ErrorType.KDC_ERR_NEVER_VALID, msg);
        }
        KerberosTime tempRtime = request.getKdcReqBody().getRTime();
        if (request.getKdcReqBody().getKdcOptions().get(27) && request.getKdcReqBody().getTill().greaterThan(kerberosEndTime)) {
            if (!config.isRenewableAllowed()) {
                String msg = "Ticket cannot be generated, as the renew date is exceeded";
                LOG_KRB.error(msg);
                throw new KerberosException(ErrorType.KDC_ERR_POLICY, msg);
            }
            request.getKdcReqBody().getKdcOptions().set(8);
            tempRtime = request.getKdcReqBody().getTill();
        }
        if (request.getKdcReqBody().getKdcOptions().get(8)) {
            if (!config.isRenewableAllowed()) {
                String msg = "Ticket cannot be generated, as Renewable is not allowed";
                LOG_KRB.error(msg);
                throw new KerberosException(ErrorType.KDC_ERR_POLICY, msg);
            }
            ticketFlags.setFlag((KerberosFlag)TicketFlag.RENEWABLE);
            if (tempRtime == null || tempRtime.isZero()) {
                tempRtime = KerberosTime.INFINITY;
            }
            long renewTill = Math.min(tempRtime.getTime(), startTime.getTime() + config.getMaximumRenewableLifetime());
            encTicketPart.setRenewTill(new KerberosTime(renewTill));
        }
        if (request.getKdcReqBody().getAddresses() != null && request.getKdcReqBody().getAddresses().getAddresses() != null && request.getKdcReqBody().getAddresses().getAddresses().length > 0) {
            encTicketPart.setClientAddresses(request.getKdcReqBody().getAddresses());
        } else if (!config.isEmptyAddressesAllowed()) {
            String msg = "Ticket cannot be generated, as the addresses are null, and it's not allowed";
            LOG_KRB.error(msg);
            throw new KerberosException(ErrorType.KDC_ERR_POLICY, msg);
        }
        EncryptedData encryptedData = cipherTextHandler.seal(serverKey, (Asn1Object)encTicketPart, KeyUsage.AS_OR_TGS_REP_TICKET_WITH_SRVKEY);
        Ticket newTicket = new Ticket(ticketPrincipal, encryptedData);
        newTicket.setRealm(serverRealm);
        newTicket.setEncTicketPart(encTicketPart);
        if (LOG_KRB.isDebugEnabled()) {
            LOG_KRB.debug("Ticket will be issued for access to {}.", (Object)serverPrincipal);
        }
        authContext.setTicket(newTicket);
    }

    private static void buildReply(AuthenticationContext authContext) throws KerberosException {
        LOG_KRB.debug("--> Building reply");
        KdcReq request = authContext.getRequest();
        Ticket ticket = authContext.getTicket();
        AsRep reply = new AsRep();
        reply.setCName(request.getKdcReqBody().getCName());
        reply.setCRealm(request.getKdcReqBody().getRealm());
        reply.setTicket(ticket);
        EncKdcRepPart encKdcRepPart = new EncKdcRepPart();
        encKdcRepPart.setKey(ticket.getEncTicketPart().getKey());
        LastReq lastReq = new LastReq();
        lastReq.addEntry(new LastReqEntry(LastReqType.TIME_OF_INITIAL_REQ, new KerberosTime()));
        encKdcRepPart.setLastReq(lastReq);
        encKdcRepPart.setNonce(request.getKdcReqBody().getNonce());
        encKdcRepPart.setFlags(ticket.getEncTicketPart().getFlags());
        encKdcRepPart.setAuthTime(ticket.getEncTicketPart().getAuthTime());
        encKdcRepPart.setStartTime(ticket.getEncTicketPart().getStartTime());
        encKdcRepPart.setEndTime(ticket.getEncTicketPart().getEndTime());
        if (ticket.getEncTicketPart().getFlags().isRenewable()) {
            encKdcRepPart.setRenewTill(ticket.getEncTicketPart().getRenewTill());
        }
        encKdcRepPart.setSName(ticket.getSName());
        encKdcRepPart.setSRealm(ticket.getRealm());
        encKdcRepPart.setClientAddresses(ticket.getEncTicketPart().getClientAddresses());
        EncAsRepPart encAsRepPart = new EncAsRepPart();
        encAsRepPart.setEncKdcRepPart(encKdcRepPart);
        if (LOG_KRB.isDebugEnabled()) {
            AuthenticationService.monitorContext(authContext);
            AuthenticationService.monitorReply(reply, encKdcRepPart);
        }
        EncryptionKey clientKey = authContext.getClientKey();
        EncryptedData encryptedData = CIPHER_TEXT_HANDLER.seal(clientKey, (Asn1Object)encAsRepPart, KeyUsage.AS_REP_ENC_PART_WITH_CKEY);
        reply.setEncPart(encryptedData);
        reply.setEncKdcRepPart(encKdcRepPart);
        authContext.setReply((KerberosMessage)reply);
    }

    private static void monitorRequest(KdcContext kdcContext) {
        KdcReq request = kdcContext.getRequest();
        if (LOG_KRB.isDebugEnabled()) {
            try {
                String clientAddress = kdcContext.getClientAddress().getHostAddress();
                StringBuilder sb = new StringBuilder();
                sb.append("Received ").append(SERVICE_NAME).append(" request:");
                sb.append("\n\tmessageType:           ").append(request.getMessageType());
                sb.append("\n\tprotocolVersionNumber: ").append(request.getProtocolVersionNumber());
                sb.append("\n\tclientAddress:         ").append(clientAddress);
                sb.append("\n\tnonce:                 ").append(request.getKdcReqBody().getNonce());
                sb.append("\n\tkdcOptions:            ").append(request.getKdcReqBody().getKdcOptions());
                sb.append("\n\tclientPrincipal:       ").append(request.getKdcReqBody().getCName());
                sb.append("\n\tserverPrincipal:       ").append(request.getKdcReqBody().getSName());
                sb.append("\n\tencryptionType:        " + KerberosUtils.getEncryptionTypesString((Set)request.getKdcReqBody().getEType()));
                sb.append("\n\trealm:                 ").append(request.getKdcReqBody().getRealm());
                sb.append("\n\tfrom time:             ").append(request.getKdcReqBody().getFrom());
                sb.append("\n\ttill time:             ").append(request.getKdcReqBody().getTill());
                sb.append("\n\trenew-till time:       ").append(request.getKdcReqBody().getRTime());
                sb.append("\n\thostAddresses:         ").append(request.getKdcReqBody().getAddresses());
                String message = sb.toString();
                LOG_KRB.debug(message);
            }
            catch (Exception e) {
                LOG_KRB.error(I18n.err((I18n)I18n.ERR_153, (Object[])new Object[0]), (Throwable)e);
            }
        }
    }

    private static void monitorContext(AuthenticationContext authContext) {
        if (LOG_KRB.isDebugEnabled()) {
            try {
                long clockSkew = authContext.getConfig().getAllowableClockSkew();
                InetAddress clientAddress = authContext.getClientAddress();
                StringBuilder sb = new StringBuilder();
                sb.append("Monitoring ").append(SERVICE_NAME).append(" context:");
                sb.append("\n\tclockSkew              ").append(clockSkew);
                sb.append("\n\tclientAddress          ").append(clientAddress);
                KerberosPrincipal clientPrincipal = authContext.getClientEntry().getPrincipal();
                PrincipalStoreEntry clientEntry = authContext.getClientEntry();
                sb.append("\n\tprincipal              ").append(clientPrincipal);
                sb.append("\n\tcn                     ").append(clientEntry.getCommonName());
                sb.append("\n\trealm                  ").append(clientEntry.getRealmName());
                sb.append("\n\tprincipal              ").append(clientEntry.getPrincipal());
                sb.append("\n\tSAM type               ").append(clientEntry.getSamType());
                PrincipalName serverPrincipal = authContext.getRequest().getKdcReqBody().getSName();
                PrincipalStoreEntry serverEntry = authContext.getServerEntry();
                sb.append("\n\tprincipal              ").append(serverPrincipal);
                sb.append("\n\tcn                     ").append(serverEntry.getCommonName());
                sb.append("\n\trealm                  ").append(serverEntry.getRealmName());
                sb.append("\n\tprincipal              ").append(serverEntry.getPrincipal());
                sb.append("\n\tSAM type               ").append(serverEntry.getSamType());
                EncryptionType encryptionType = authContext.getEncryptionType();
                int clientKeyVersion = ((EncryptionKey)clientEntry.getKeyMap().get(encryptionType)).getKeyVersion();
                int serverKeyVersion = ((EncryptionKey)serverEntry.getKeyMap().get(encryptionType)).getKeyVersion();
                sb.append("\n\tRequest key type       ").append(encryptionType);
                sb.append("\n\tClient key version     ").append(clientKeyVersion);
                sb.append("\n\tServer key version     ").append(serverKeyVersion);
                String message = sb.toString();
                LOG_KRB.debug(message);
            }
            catch (Exception e) {
                LOG_KRB.error(I18n.err((I18n)I18n.ERR_154, (Object[])new Object[0]), (Throwable)e);
            }
        }
    }

    private static void monitorReply(AsRep reply, EncKdcRepPart part) {
        if (LOG_KRB.isDebugEnabled()) {
            try {
                StringBuilder sb = new StringBuilder();
                sb.append("Responding with ").append(SERVICE_NAME).append(" reply:");
                sb.append("\n\tmessageType:           ").append(reply.getMessageType());
                sb.append("\n\tprotocolVersionNumber: ").append(reply.getProtocolVersionNumber());
                sb.append("\n\tnonce:                 ").append(part.getNonce());
                sb.append("\n\tclientPrincipal:       ").append(reply.getCName());
                sb.append("\n\tclient realm:          ").append(reply.getCRealm());
                sb.append("\n\tserverPrincipal:       ").append(part.getSName());
                sb.append("\n\tserver realm:          ").append(part.getSRealm());
                sb.append("\n\tauth time:             ").append(part.getAuthTime());
                sb.append("\n\tstart time:            ").append(part.getStartTime());
                sb.append("\n\tend time:              ").append(part.getEndTime());
                sb.append("\n\trenew-till time:       ").append(part.getRenewTill());
                sb.append("\n\thostAddresses:         ").append(part.getClientAddresses());
                String message = sb.toString();
                LOG_KRB.debug(message);
            }
            catch (Exception e) {
                LOG_KRB.error(I18n.err((I18n)I18n.ERR_155, (Object[])new Object[0]), (Throwable)e);
            }
        }
    }

    private static byte[] preparePreAuthenticationError(EncryptionType requestedType, Set<EncryptionType> encryptionTypes) {
        boolean isNewEtype = KerberosUtils.isNewEncryptionType((EncryptionType)requestedType);
        ETypeInfo2 eTypeInfo2 = new ETypeInfo2();
        ETypeInfo eTypeInfo = new ETypeInfo();
        for (EncryptionType encryptionType : encryptionTypes) {
            if (!isNewEtype) {
                ETypeInfoEntry etypeInfoEntry = new ETypeInfoEntry(encryptionType, null);
                eTypeInfo.addETypeInfoEntry(etypeInfoEntry);
            }
            ETypeInfo2Entry etypeInfo2Entry = new ETypeInfo2Entry(encryptionType);
            eTypeInfo2.addETypeInfo2Entry(etypeInfo2Entry);
        }
        byte[] encTypeInfo = null;
        byte[] encTypeInfo2 = null;
        try {
            ByteBuffer buffer;
            if (!isNewEtype) {
                buffer = ByteBuffer.allocate(eTypeInfo.computeLength());
                encTypeInfo = eTypeInfo.encode(buffer).array();
            }
            buffer = ByteBuffer.allocate(eTypeInfo2.computeLength());
            encTypeInfo2 = eTypeInfo2.encode(buffer).array();
        }
        catch (EncoderException ioe) {
            return null;
        }
        MethodData methodData = new MethodData();
        methodData.addPaData(new PaData(PaDataType.PA_ENC_TIMESTAMP, null));
        if (!isNewEtype) {
            methodData.addPaData(new PaData(PaDataType.PA_ENCTYPE_INFO, encTypeInfo));
        }
        methodData.addPaData(new PaData(PaDataType.PA_ENCTYPE_INFO2, encTypeInfo2));
        try {
            ByteBuffer buffer = ByteBuffer.allocate(methodData.computeLength());
            return methodData.encode(buffer).array();
        }
        catch (EncoderException ee) {
            LOG_KRB.warn("Failed to encode the etype information", (Throwable)ee);
            return null;
        }
    }
}

